package com.usercentrics.sdk.ui.banner

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.annotation.ColorInt
import androidx.transition.*
import com.usercentrics.sdk.ui.theme.UCThemeData

internal interface UCBannerTransition {
    val rootView: View
    fun enter()
    fun exit(callback: () -> Unit)
}

internal class UCBannerTransitionImpl(
    private val context: Context,
    private val theme: UCThemeData,
    @ColorInt private val customOverlayColor: Int?,
    private var bannerContainerView: UCBannerContainerView,
    private val slideTransitionEnabled: Boolean,
) : UCBannerTransition {

    companion object {
        private const val defaultAnimationDuration = 300L
    }

    private val backgroundOverlayColor: Int?
        get() = customOverlayColor ?: theme.colorPalette.overlayColor

    private val dialogBackgroundView: FrameLayout by lazy {
        FrameLayout(context).apply {
            visibility = View.INVISIBLE
            layoutParams = FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)

            addView(bannerContainerView)
            backgroundOverlayColor?.let { setBackgroundColor(it) }
        }
    }

    override val rootView: View by lazy { dialogBackgroundView }

    override fun enter() {
        if (slideTransitionEnabled) {
            slideDialogUp()
        } else {
            dialogBackgroundView.visibility = View.VISIBLE
            bannerContainerView.visibility = View.VISIBLE
        }
    }

    override fun exit(callback: () -> Unit) {
        if (slideTransitionEnabled) {
            slideDialogDown(callback)
        } else {
            callback()
        }
    }

    private fun slideDialogUp() {
        Handler(Looper.getMainLooper()).post {
            slideDialog(BannerTransitionParameters.SlideUp)
        }
    }

    private fun slideDialogDown(callback: () -> Unit) {
        slideDialog(BannerTransitionParameters.SlideDown, callback)
    }

    private fun slideDialog(parameters: BannerTransitionParameters, callback: (() -> Unit)? = null) {
        val slideTransition = Slide(parameters.gravity).apply {
            duration = defaultAnimationDuration
            addTarget(bannerContainerView)
        }

        TransitionManager.beginDelayedTransition(bannerContainerView as ViewGroup, slideTransition)
        bannerContainerView.visibility = parameters.visibility

        val fadeTransition = Fade(parameters.fadingMode).apply {
            duration = defaultAnimationDuration
            addTarget(dialogBackgroundView)
        }

        if (callback != null) {
            fadeTransition.addListener(object : Transition.TransitionListener {
                override fun onTransitionEnd(transition: Transition) {
                    callback()
                }

                override fun onTransitionStart(transition: Transition) {}
                override fun onTransitionCancel(transition: Transition) {}
                override fun onTransitionPause(transition: Transition) {}
                override fun onTransitionResume(transition: Transition) {}
            })
        }

        TransitionManager.beginDelayedTransition(dialogBackgroundView, fadeTransition)
        dialogBackgroundView.visibility = parameters.visibility
    }
}

private sealed class BannerTransitionParameters(
    @Slide.GravityFlag val gravity: Int,
    @Visibility.Mode val fadingMode: Int,
    val visibility: Int
) {
    object SlideUp : BannerTransitionParameters(Gravity.BOTTOM, Fade.IN, View.VISIBLE)
    object SlideDown : BannerTransitionParameters(Gravity.BOTTOM, Fade.OUT, View.INVISIBLE)
}
