package com.usercentrics.sdk.ui.components

import android.content.Context
import android.graphics.drawable.Drawable
import android.os.Build
import android.text.TextUtils
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import androidx.annotation.AttrRes
import androidx.constraintlayout.widget.ConstraintLayout
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.accessibility.accessibleTouchTarget
import com.usercentrics.sdk.ui.components.cards.UCControllerIdPM
import com.usercentrics.sdk.ui.components.cards.getCardBackground
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getCheckOutlinedCircleIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getCopyControllerIdIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.styleIcon
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class UCControllerId : ConstraintLayout {

    companion object {
        private const val COPIED_DURATION_MILLIS = 3500L
    }

    private val ucControllerIdLabel: UCTextView by lazy { findViewById(R.id.ucControllerIdLabel) }
    private val ucControllerIdValue: UCTextView by lazy { findViewById(R.id.ucControllerIdValue) }
    private val ucControllerIdCopy: UCImageView by lazy { findViewById(R.id.ucControllerIdCopy) }

    private val defaultIconDrawable: Drawable? by lazy { context.getCopyControllerIdIcon() }
    private val checkedIconDrawable: Drawable? by lazy { context.getCheckOutlinedCircleIcon() }

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initLayout(context)
    }

    private fun initLayout(context: Context) {
        LayoutInflater.from(context).inflate(R.layout.uc_controller_id, this)
        setIconDefaultState()
    }

    fun style(theme: UCThemeData) {
        background = theme.colorPalette.getCardBackground(context)

        ucControllerIdLabel.styleSmall(theme, isSecondary = true)
        ucControllerIdValue.styleBody(theme)
        defaultIconDrawable?.styleIcon(theme)
        checkedIconDrawable?.styleIcon(theme)
    }

    fun bind(model: UCControllerIdPM) {
        ucControllerIdLabel.text = model.label

        ucControllerIdValue.apply {
            text = model.value
            importantForAccessibility = IMPORTANT_FOR_ACCESSIBILITY_NO

            val isRtl = resources.configuration.layoutDirection == View.LAYOUT_DIRECTION_RTL
            ellipsize = if (isRtl) {
                TextUtils.TruncateAt.START
            } else {
                TextUtils.TruncateAt.END
            }
        }

        ucControllerIdCopy.apply {
            contentDescription = model.ariaLabel

            setOnClickListener {
                model.onCopyControllerId()

                setIconCheckedState()

                postDelayed({ setIconDefaultState() }, COPIED_DURATION_MILLIS)
            }
        }
    }

    private fun setIconCheckedState() {
        ucControllerIdCopy.apply {
            isEnabled = false
            setImageDrawable(checkedIconDrawable)
        }
    }

    private fun setIconDefaultState() {
        ucControllerIdCopy.apply {
            isEnabled = true
            setImageDrawable(defaultIconDrawable)
            accessibleTouchTarget()
        }
    }
}
