package com.usercentrics.sdk.ui.components

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.Paint
import android.graphics.Typeface
import android.text.SpannableString
import android.text.Spanned
import android.text.TextPaint
import android.text.style.ClickableSpan
import android.text.style.URLSpan
import android.util.AttributeSet
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import androidx.annotation.AttrRes
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.text.HtmlCompat
import com.usercentrics.sdk.models.settings.PredefinedUIHtmlLinkType
import com.usercentrics.sdk.ui.theme.UCThemeData

open class UCTextView : AppCompatTextView {
    companion object {
        private const val defaultUnderlineLink = true
    }

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(
        context: Context,
        attrs: AttributeSet?,
        @AttrRes defStyleAttr: Int
    ) : super(context, attrs, defStyleAttr)

    fun setHtmlText(
        htmlText: String,
        underlineLink: Boolean? = null,
        predefinedUILinkHandler: (PredefinedUIHtmlLinkType) -> Unit,
    ) {
        val spannedHtml = HtmlCompat.fromHtml(htmlText, HtmlCompat.FROM_HTML_MODE_LEGACY)
        val spannableString = SpannableString(spannedHtml)
        this.text = processHtml(spannableString, underlineLink, predefinedUILinkHandler)
    }

    fun setHtmlTextWithNoLinks(htmlText: String) {
        val spannedHtml = HtmlCompat.fromHtml(htmlText, HtmlCompat.FROM_HTML_MODE_LEGACY)
        val spannableString = SpannableString(spannedHtml)
        this.text = processHtmlWithNoLinks(spannableString)
    }

    private fun processHtmlWithNoLinks(spannableString: SpannableString): CharSequence {
        val urlSpans = spannableString.getSpans(0, spannableString.length, URLSpan::class.java)
        urlSpans.forEach {
            spannableString.removeSpan(it)
        }
        return spannableString
    }

    private fun processHtml(
        spannableString: SpannableString,
        underlineLink: Boolean?,
        predefinedUILinkHandler: (PredefinedUIHtmlLinkType) -> Unit,
    ): CharSequence {
        val shouldUnderlineLinks = underlineLink ?: defaultUnderlineLink
        val urlSpans = spannableString.getSpans(0, spannableString.length, URLSpan::class.java)

        urlSpans.forEach { urlSpan ->
            val linkType = PredefinedUIHtmlLinkType.from(urlSpan.url)

            if (linkType != null) {
                val replacement = PredefinedUILinkSpan(linkType, predefinedUILinkHandler, shouldUnderlineLinks)
                interceptLink(spannableString, urlSpan, replacement)
            } else {
                val isJavaScriptCode = urlSpan.url.startsWith("javascript:UC_UI")

                if (isJavaScriptCode) {
                    spannableString.removeSpan(urlSpan)
                } else {
                    val replacement = ExternalLinkSpan(urlSpan.url, shouldUnderlineLinks)
                    interceptLink(spannableString, urlSpan, replacement)
                }
            }
        }

        return spannableString
    }


    private fun interceptLink(spannableString: SpannableString, targetSpan: URLSpan, replacement: ClickableSpan) {
        val start = spannableString.getSpanStart(targetSpan)
        val end = spannableString.getSpanEnd(targetSpan)
        spannableString.removeSpan(targetSpan)
        spannableString.setSpan(replacement, start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE)
    }

    private class PredefinedUILinkSpan(
        private val link: PredefinedUIHtmlLinkType,
        private val handler: (PredefinedUIHtmlLinkType) -> Unit,
        private val isUnderlineText: Boolean
    ) : ClickableSpan() {

        override fun onClick(widget: View) = handler(link)

        override fun updateDrawState(ds: TextPaint) {
            super.updateDrawState(ds)
            ds.isUnderlineText = isUnderlineText
        }
    }

    private class ExternalLinkSpan(
        url: String,
        private val isUnderlineText: Boolean
    ) : URLSpan(url) {
        override fun updateDrawState(ds: TextPaint) {
            super.updateDrawState(ds)
            ds.isUnderlineText = isUnderlineText
        }
    }

    fun styleTitle(theme: UCThemeData) {
        setTypeface(theme.fonts.font, Typeface.BOLD)
        theme.colorPalette.text100?.let { setTextColor(it) }
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.title)
        paintFlags = Paint.ANTI_ALIAS_FLAG
    }

    fun styleBody(
        theme: UCThemeData,
        isBold: Boolean = false,
        isLink: Boolean = false,
        isSecondary: Boolean = false
    ) {
        if (isBold) {
            setTypeface(theme.fonts.font, Typeface.BOLD)
        } else {
            typeface = theme.fonts.font
        }

        when {
            isLink -> theme.colorPalette.linkColor
            isSecondary -> theme.colorPalette.text80
            else -> theme.colorPalette.text100
        }?.let { setTextColor(it) }
        theme.colorPalette.linkColor?.let { setLinkTextColor(it) }
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.body)
        paintFlags = Paint.ANTI_ALIAS_FLAG
    }

    fun styleSmall(
        theme: UCThemeData,
        isUnderline: Boolean = false,
        isBold: Boolean = false,
        isSecondary: Boolean = false,
        isLink: Boolean = false
    ) {
        if (isBold) {
            setTypeface(theme.fonts.font, Typeface.BOLD)
        } else {
            typeface = theme.fonts.font
        }
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.small)
        when {
            isLink -> theme.colorPalette.linkColor
            isSecondary -> theme.colorPalette.text80
            else -> theme.colorPalette.text100
        }?.let { setTextColor(it) }
        paintFlags = if (isUnderline) Paint.UNDERLINE_TEXT_FLAG or Paint.ANTI_ALIAS_FLAG
        else Paint.ANTI_ALIAS_FLAG
    }

    fun styleTiny(theme: UCThemeData) {
        typeface = theme.fonts.font
        theme.colorPalette.text80?.let { setTextColor(it) }
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.tiny)
        paintFlags = Paint.ANTI_ALIAS_FLAG
    }

    fun styleTab(theme: UCThemeData) {
        val states = arrayOf(
            intArrayOf(android.R.attr.state_selected),
            intArrayOf(-android.R.attr.state_selected)
        )

        if (theme.colorPalette.selectedTabColor != null && theme.colorPalette.text100 != null) {
            val colors = intArrayOf(
                theme.colorPalette.selectedTabColor,
                theme.colorPalette.text100
            )
            val colorList = ColorStateList(states, colors)
            setTextColor(colorList)
        }

        isAllCaps = false
        gravity = Gravity.CENTER_HORIZONTAL
        typeface = theme.fonts.font
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.body)
    }

    fun styleSelectedTab(theme: UCThemeData) {
        setTypeface(theme.fonts.font, Typeface.BOLD)
    }

    fun styleUnselectedTab(theme: UCThemeData) {
        typeface = theme.fonts.font
    }

    fun styleSectionTitle(theme: UCThemeData) {
        setTypeface(theme.fonts.font, Typeface.BOLD)
        setTextSize(TypedValue.COMPLEX_UNIT_SP, theme.fonts.sizes.body)
        theme.colorPalette.text100?.let { setTextColor(it) }
        paintFlags = Paint.ANTI_ALIAS_FLAG
    }
}
