package com.usercentrics.sdk.ui.extensions

import android.app.Activity
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.provider.Browser
import android.util.TypedValue
import android.view.LayoutInflater
import androidx.appcompat.view.ContextThemeWrapper
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.R

internal val Context.inflater: LayoutInflater get() = LayoutInflater.from(this)

internal fun Context.copyToClipboard(text: String, clipboardLabel: String = ""): Boolean {
    return try {
        val clipboard = getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        val clip = ClipData.newPlainText(clipboardLabel, text)
        clipboard.setPrimaryClip(clip)
        true
    } catch (e: Throwable) {
        false
    }
}

internal fun Context.themed(): Context {
    return ContextThemeWrapper(this, R.style.BaseTheme)
}

internal fun Context.openUrl(url: String) {
    try {
        assert(url.isNotBlank()) { "Empty URL" }
        startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url.addHttpsIfNeeded())).putExtra(Browser.EXTRA_APPLICATION_ID, packageName))
    } catch (e: Throwable) {
        PredefinedUIDependencyManager.logger.error("Error when opening URL<$url>", e)
    }
}

internal fun Context.safeShowBanner(showBannerHandler: () -> Unit) {
    if (isActivityFinishingOrDestroyed()) {
        return
    }

    try {
        showBannerHandler()
    } catch (cause: Exception) {
        assert(false) { "Show Banner Error: $cause".also { cause.printStackTrace() } }
    }
}

internal fun Context.isActivityFinishingOrDestroyed(): Boolean {
    // This is not a silver bullet since the context could be a wrapper of an invalid Activity
    val activity = this as? Activity
    return activity == null || activity.isFinishing || activity.isDestroyed
}

// This is a static object that forward the extensions that we use from the TV-UI module.
// We do not want to expose extensions methods with common names that could confuse the customer IDE.
object PredefinedUIContextExtensions {
    fun themed(context: Context): Context = context.themed()
    fun safeShowBanner(context: Context, showBannerHandler: () -> Unit) = context.safeShowBanner(showBannerHandler)
}

internal fun Context.dpToPx(value: Int) = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, value.toFloat(), resources.displayMetrics)

