package com.usercentrics.sdk.ui.firstLayer

import com.usercentrics.sdk.*
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.predefinedUI.PredefinedUIConsentManager
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.banner.SecondLayerInitialState
import com.usercentrics.sdk.ui.banner.UCBannerCoordinator
import com.usercentrics.sdk.ui.components.UCButtonSettings
import com.usercentrics.sdk.ui.components.UCButtonType
import com.usercentrics.sdk.ui.extensions.emptyToNull
import com.usercentrics.sdk.ui.mappers.UCCategoryMapperImpl
import com.usercentrics.sdk.ui.secondLayer.UCCardsContentPM
import com.usercentrics.sdk.ui.theme.UCThemeData
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleGroup
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediator

internal class UCFirstLayerViewModelImpl(
    private val layout: UsercentricsLayout,
    private val layerSettings: UCUIFirstLayerSettings,
    private val consentManager: PredefinedUIConsentManager?,
    private val buttonLabels: FirstLayerButtonLabels,
    private val theme: UCThemeData,
    settings: FirstLayerStyleSettings?,
    generalLogo: UsercentricsImage?,
    private val linksSettings: LegalLinksSettings,
    private val coordinator: UCBannerCoordinator,
    private val toggleMediator: PredefinedUIToggleMediator,
    private val landscapeMode: Boolean,
    override val ariaLabels: PredefinedUIAriaLabels,
    override val statusBarColor: Int?
) : UCFirstLayerViewModel {

    private var ccpaToggleValue: Boolean? = layerSettings.footerSettings.optOutToggleInitialValue

    override val closeLink: String?
        get() = layerSettings.headerSettings.firstLayerCloseLink

    override val closeIcon: Boolean
        get() = layerSettings.headerSettings.firstLayerCloseIcon ?: false

    override val shortDescription
        get() = (layerSettings.headerSettings as PredefinedUIFirstLayerHeaderSettings).shortDescription

    override val readMore: String? by lazy {
        if (shortDescriptionMessage == null) {
            null
        } else {
            (layerSettings.headerSettings as PredefinedUIFirstLayerHeaderSettings).readMoreText
        }
    }

    override val headerImage: HeaderImageSettings? by lazy {
        val logoFromAI by lazy { UsercentricsImage.ImageUrl.from(layerSettings.headerSettings.logoURL) }
        val alignmentFromAI by lazy { SectionAlignment.from(layerSettings.headerSettings.logoPosition) }

        return@lazy when {
            settings?.headerImage != null -> settings.headerImage
            generalLogo != null -> {
                HeaderImageSettings.LogoSettings(
                    image = generalLogo,
                    alignment = alignmentFromAI
                )
            }

            logoFromAI != null -> {
                HeaderImageSettings.LogoSettings(
                    image = logoFromAI!!,
                    alignment = alignmentFromAI
                )
            }

            else -> {
                null
            }
        }
    }

    override val buttons: List<List<UCButtonSettings>> by lazy {
        UCButtonSettings.map(
            isCCPA = ccpaToggle != null,
            buttonLayout = settings?.buttonLayout,
            defaultButtons = defaultButtons,
            theme = theme,
            buttonLabels = buttonLabels,
        )
    }

    private val defaultButtons: List<List<PredefinedUIFooterButton>> by lazy {
        if (landscapeMode) {
            layerSettings.footerSettings.buttonsLandscape
        } else {
            layerSettings.footerSettings.buttons
        }
    }

    override val title: UCFirstLayerTitle? by lazy {
        val text = layerSettings.headerSettings.title
        UCFirstLayerTitle(
            text = text,
            customFont = settings?.title?.font,
            customTextSizeInSp = settings?.title?.textSizeInSp,
            customTextColor = settings?.title?.textColor,
            customAlignment = settings?.title?.alignment
        )
    }

    override val useAllAvailableVerticalSpace: Boolean by lazy { layout is UsercentricsLayout.Full }

    private val shortDescriptionMessage: UCFirstLayerMessage? by lazy {
        val shortDescription = shortDescription

        if (shortDescription == null) {
            null
        } else {
            UCFirstLayerMessage(
                text = shortDescription,
                customFont = settings?.message?.font,
                customTextSizeInSp = settings?.message?.textSizeInSp,
                customAlignment = settings?.message?.alignment,
                customTextColor = settings?.message?.textColor,
                customLinkTextColor = settings?.message?.linkTextColor ?: settings?.message?.textColor,
                customUnderlineLink = settings?.message?.underlineLink
            )
        }
    }

    private val contentDescriptionMessage: UCFirstLayerMessage by lazy {
        UCFirstLayerMessage(
            text = layerSettings.headerSettings.contentDescription,
            customFont = settings?.message?.font,
            customTextSizeInSp = settings?.message?.textSizeInSp,
            customAlignment = settings?.message?.alignment,
            customTextColor = settings?.message?.textColor,
            customLinkTextColor = settings?.message?.linkTextColor ?: settings?.message?.textColor,
            customUnderlineLink = settings?.message?.underlineLink
        )
    }

    private var readMoreClicked = false

    override val message: UCFirstLayerMessage
        get() {
            return if (shortDescriptionMessage == null) {
                contentDescriptionMessage
            } else if (readMoreClicked) {
                contentDescriptionMessage
            } else {
                shortDescriptionMessage!!
            }
        }

    override val legalLinks: List<PredefinedUILink>? by lazy {
        val filterLegalLinks = linksSettings == LegalLinksSettings.SECOND_LAYER_ONLY || linksSettings == LegalLinksSettings.HIDDEN
        layerSettings.headerSettings.links.filterNot {
            filterLegalLinks && it.linkType == PredefinedUILinkType.URL
        }.emptyToNull()
    }

    override val poweredByLabel: String?
        get() = layerSettings.footerSettings.poweredBy?.label

    override val ccpaToggle: UCFirstLayerCCPAToggle? by lazy {
        val optOutToggle = layerSettings.footerSettings.optOutToggle ?: return@lazy null
        UCFirstLayerCCPAToggle(
            label = optOutToggle.label,
            initialValue = layerSettings.footerSettings.optOutToggleInitialValue
        )
    }

    override val content: List<UCCardsContentPM>? by lazy {
        val mapper = UCCategoryMapperImpl()
        val result = layerSettings.contentSettings.map { cardUISection ->
            UCCardsContentPM(
                title = cardUISection.title,
                cards = cardUISection.cards.map { category ->
                    val mainGroup = category.createToggleGroup()
                    mapper.map(category, mainGroup, toggleMediator)
                }.also {
                    toggleMediator.bootLegacy()
                },
                controllerId = null
            )
        }
        result.emptyToNull()
    }

    private fun PredefinedUICardUI.createToggleGroup(): PredefinedUIToggleGroup? {
        return toggleMediator.getGroupLegacy(this)
    }

    override fun onCCPAToggleChanged(state: Boolean) {
        ccpaToggleValue = state

        val event = if (state) {
            UsercentricsAnalyticsEventType.CCPA_TOGGLES_ON
        } else {
            UsercentricsAnalyticsEventType.CCPA_TOGGLES_OFF
        }
        trackAnalyticsEvent(event)
    }

    override fun onHtmlLinkClick(type: PredefinedUIHtmlLinkType) {
        when (type) {
            PredefinedUIHtmlLinkType.ACCEPT_ALL_LINK -> onAcceptAllClick()
            PredefinedUIHtmlLinkType.DENY_ALL_LINK -> onDenyAllClick()
            PredefinedUIHtmlLinkType.SHOW_SECOND_LAYER -> onMoreClick()
        }
    }

    override fun onLegalLinkClick(link: PredefinedUILink) {
        when (link.linkType) {
            PredefinedUILinkType.URL -> onHyperlinkClick(link)
            PredefinedUILinkType.MANAGE_SETTINGS -> onMoreClick(link.linkType.tabIndex)
            PredefinedUILinkType.VENDOR_LIST -> onMoreClick(link.linkType.tabIndex)
        }
    }

    private fun onHyperlinkClick(link: PredefinedUILink) {
        coordinator.navigateToUrl(link.url)
        trackAnalyticsEvent(link.eventType)
    }

    override fun onButtonClick(type: UCButtonType) {
        when (type) {
            UCButtonType.ACCEPT_ALL -> onAcceptAllClick()
            UCButtonType.DENY_ALL -> onDenyAllClick()
            UCButtonType.MORE -> onMoreClick()
            UCButtonType.OK -> onOkClick()
            UCButtonType.SAVE -> onSaveClick()
        }
    }

    override fun onReadMoreClick() {
        readMoreClicked = true
    }

    private fun onAcceptAllClick() {
        val feedback = consentManager?.acceptAll(TCFDecisionUILayer.FIRST_LAYER)
        coordinator.finishCMP(feedback?.toUserResponse())
    }

    private fun onDenyAllClick() {
        val feedback = consentManager?.denyAll(TCFDecisionUILayer.FIRST_LAYER)
        coordinator.finishCMP(feedback?.toUserResponse())
    }

    private fun onMoreClick(initialTabIndex: Int? = null) {
        coordinator.navigateToSecondLayer(
            SecondLayerInitialState(
                ccpaToggleValue = ccpaToggleValue,
                tabIndex = initialTabIndex,
            )
        )
        trackAnalyticsEvent(UsercentricsAnalyticsEventType.MORE_INFORMATION_LINK)
    }

    private fun trackAnalyticsEvent(event: UsercentricsAnalyticsEventType) {
        PredefinedUIDependencyManager.analyticsManager.track(event)
    }

    private fun onOkClick() {
        val feedback = if (ccpaToggleValue == true) {
            consentManager?.denyAll(TCFDecisionUILayer.FIRST_LAYER)
        } else {
            consentManager?.acceptAll(TCFDecisionUILayer.FIRST_LAYER)
        }
        coordinator.finishCMP(feedback?.toUserResponse())
    }

    private fun onSaveClick() {
        val feedback = consentManager?.save(
            TCFDecisionUILayer.FIRST_LAYER,
            toggleMediator.getUserDecisions(),
        )
        coordinator.finishCMP(feedback?.toUserResponse())
    }
}
