package com.usercentrics.sdk.ui.firstLayer.component

import android.graphics.Paint
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.text.TextUtils
import android.util.TypedValue
import android.view.Gravity
import androidx.appcompat.widget.LinearLayoutCompat
import androidx.core.view.ViewCompat
import com.usercentrics.sdk.models.settings.PredefinedUIHtmlLinkType
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCImageView
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getBackButtonIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getCloseIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.styleIcon
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerViewModel
import com.usercentrics.sdk.ui.theme.UCThemeData

internal fun LinearLayoutCompat.addCloseButton(theme: UCThemeData, viewModel: UCFirstLayerViewModel) {
    val icon = context.getCloseIcon()?.also { it.styleIcon(theme) }
    viewModel.message.customTextColor?.let {
        icon?.colorFilter = PorterDuffColorFilter(it, PorterDuff.Mode.SRC_IN)
    }

    val closeIcon = UCImageView(context).apply {
        val padding = 13.dpToPx(context)
        setPaddingRelative(padding, padding, padding, padding)

        setOnClickListener { viewModel.onHtmlLinkClick(PredefinedUIHtmlLinkType.DENY_ALL_LINK) }
        setImageDrawable(icon)

        setBackgroundResource(TypedValue().apply {
            val resId = android.R.attr.selectableItemBackgroundBorderless
            context.theme.resolveAttribute(resId, this, true)
        }.resourceId)

        contentDescription = viewModel.ariaLabels.closeButton
    }

    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.WRAP_CONTENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT).apply {
        gravity = Gravity.END

        marginStart = 0
        marginEnd = 4.dpToPx(context)
    }
    addView(closeIcon, layoutParams)
}

internal fun LinearLayoutCompat.addContinueWithoutAccepting(theme: UCThemeData, viewModel: UCFirstLayerViewModel) {
    val container = LinearLayoutCompat(context).apply {
        orientation = LinearLayoutCompat.HORIZONTAL
        setBackgroundResource(TypedValue().apply {
            context.theme.resolveAttribute(android.R.attr.selectableItemBackground, this, true)
        }.resourceId)

        val margins = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)
        setPaddingRelative(margins, margins, margins, margins)

        setOnClickListener { viewModel.onHtmlLinkClick(PredefinedUIHtmlLinkType.DENY_ALL_LINK) }
    }

    // Link text
    val closeLinkView = UCTextView(context).apply {
        text = viewModel.closeLink
        styleSmall(theme, isLink = true)
        includeFontPadding = false
        if (viewModel.message.customUnderlineLink == true) {
            paintFlags = paintFlags or Paint.UNDERLINE_TEXT_FLAG
        }
    }
    viewModel.message.customTextColor?.let {
        closeLinkView.setTextColor(it)
    }
    viewModel.message.customLinkTextColor?.let {
        closeLinkView.setTextColor(it)
    }
    container.addView(closeLinkView, LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.WRAP_CONTENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT))

    // Link icon
    val icon = context.getBackButtonIcon()?.also {
        it.styleIcon(theme)
    }
    viewModel.message.customLinkTextColor?.let {
        icon?.colorFilter = PorterDuffColorFilter(it, PorterDuff.Mode.SRC_IN)
    }
    val iconView = UCImageView(context).apply {
        setImageDrawable(icon)
        val isRightToLeft = TextUtils.getLayoutDirectionFromLocale(closeLinkView.textLocale) == ViewCompat.LAYOUT_DIRECTION_RTL
        rotationY = if (isRightToLeft) 0f else 180f
    }
    container.addView(iconView, LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.WRAP_CONTENT, LinearLayoutCompat.LayoutParams.MATCH_PARENT))

    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.WRAP_CONTENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT).apply {
        gravity = Gravity.END
    }
    addView(container, layoutParams)
}
