package com.usercentrics.sdk.ui.firstLayer.component

import android.graphics.Rect
import android.text.method.LinkMovementMethod
import android.util.TypedValue
import android.view.TouchDelegate
import androidx.appcompat.widget.LinearLayoutCompat
import com.usercentrics.sdk.SectionAlignment.Companion.toGravity
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.accessibility.createAccessibilityLinkOrButton
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerViewModel
import com.usercentrics.sdk.ui.theme.UCThemeData


internal fun LinearLayoutCompat.addMessageAndReadMoreButton(theme: UCThemeData, viewModel: UCFirstLayerViewModel) {
    val messageView = createMessageView(theme, viewModel)
    createOrRemoveReadMoreView(theme, viewModel, messageView)
}

private fun LinearLayoutCompat.createMessageView(theme: UCThemeData, viewModel: UCFirstLayerViewModel): UCTextView {
    val messageView = UCTextView(context).apply {
        setHtmlText(
            htmlText = viewModel.message.text,
            underlineLink = viewModel.message.customUnderlineLink,
            predefinedUILinkHandler = viewModel::onHtmlLinkClick
        )
        styleBody(theme)
        movementMethod = LinkMovementMethod.getInstance()
    }

    setMessageView(viewModel, messageView)

    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT).apply {
        val horizontalMargin = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)

        marginStart = horizontalMargin
        marginEnd = horizontalMargin
    }

    addView(messageView, layoutParams)
    return messageView
}

private fun updateMessageView(viewModel: UCFirstLayerViewModel, messageView: UCTextView) {
    messageView.setHtmlText(
        htmlText = viewModel.message.text,
        underlineLink = viewModel.message.customUnderlineLink,
        predefinedUILinkHandler = viewModel::onHtmlLinkClick
    )

    setMessageView(viewModel, messageView)
}

private fun LinearLayoutCompat.createOrRemoveReadMoreView(theme: UCThemeData, viewModel: UCFirstLayerViewModel, messageView: UCTextView) {
    val readMore = viewModel.readMore ?: return

    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT).apply {
        val horizontalMargin = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)
        marginStart = horizontalMargin
        marginEnd = horizontalMargin
    }

    val readMoreView = UCTextView(context)
    addView(readMoreView, layoutParams)

    readMoreView.apply {
        text = readMore
        styleSmall(theme = theme, isUnderline = true, isLink = true)
        createAccessibilityLinkOrButton()
    }

    readMoreView.setOnClickListener {
        viewModel.onReadMoreClick()
        removeView(readMoreView)
        updateMessageView(viewModel, messageView)
    }
}

private fun setMessageView(viewModel: UCFirstLayerViewModel, messageView: UCTextView) {
    viewModel.message.customAlignment?.let {
        messageView.gravity = it.toGravity()
    }
    viewModel.message.customFont?.let {
        messageView.typeface = it
    }
    viewModel.message.customTextSizeInSp?.let {
        messageView.setTextSize(TypedValue.COMPLEX_UNIT_SP, it)
    }
    viewModel.message.customTextColor?.let {
        messageView.setTextColor(it)
    }
    viewModel.message.customLinkTextColor?.let {
        messageView.setLinkTextColor(it)
    }
}
