package com.usercentrics.sdk.ui.mappers

import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.ui.components.UCTogglePM
import com.usercentrics.sdk.ui.components.cards.UCCardPM
import com.usercentrics.sdk.ui.components.cards.UCContentSectionPM
import com.usercentrics.sdk.ui.components.cards.UCContentTextSectionPM
import com.usercentrics.sdk.ui.components.cards.UContentToggleEntryPM
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleGroup
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediator

internal interface UCCategoryMapper {

    fun map(
        category: PredefinedUICardUI,
        mainGroup: PredefinedUIToggleGroup?,
        toggleMediator: PredefinedUIToggleMediator,
    ): UCCardPM
}

internal class UCCategoryMapperImpl : UCCategoryMapper {

    override fun map(
        category: PredefinedUICardUI,
        mainGroup: PredefinedUIToggleGroup?,
        toggleMediator: PredefinedUIToggleMediator,
    ): UCCardPM {

        val switchSettings = category.mainSwitchSettings
        val mainToggle = if (switchSettings != null) {
            UCTogglePM(switchSettings, group = mainGroup)
        } else {
            null
        }

        val toggleList = category.switchSettings?.map { itemSwitchSettings ->
            val group = toggleMediator.getServiceGroupLegacy(category.id, itemSwitchSettings)
            UCTogglePM(itemSwitchSettings, group)
        }

        return UCCardPM(
            id = category.id,
            title = category.title,
            description = category.shortDescription,
            mainToggle = mainToggle,
            contentSections = categoryContentSections(category, toggleMediator),
            toggleList = toggleList
        )
    }

    private fun categoryContentSections(category: PredefinedUICardUI, toggleMediator: PredefinedUIToggleMediator): List<UCContentSectionPM> {
        return when (val cardContent = category.content) {
            is PredefinedUIServicesCardContent -> mapServicesCardContent(cardContent, toggleMediator)
            is PredefinedUISimpleCardContent -> mapSimpleCardContent(cardContent)
            is PredefinedUIPurposeCardContent -> mapPurposesCardContent(cardContent)
            else -> listOf()
        }
    }

    private fun mapServicesCardContent(servicesCardContent: PredefinedUIServicesCardContent, toggleMediator: PredefinedUIToggleMediator): List<UContentToggleEntryPM> {
        return servicesCardContent.services.map { serviceDetails ->
            mapServiceContent(serviceDetails, serviceDetails.mainSwitchSettings, toggleMediator)
        }
    }

    private fun mapPurposesCardContent(purposesCardContent: PredefinedUIPurposeCardContent): List<UCContentSectionPM> {
        return mapSimpleCardContent(purposesCardContent.examples) + mapVendorsContent(purposesCardContent.vendors)
    }

    private fun mapVendorsContent(simpleCardContent: PredefinedUIPurposeVendorDetails): List<UCContentSectionPM> {
        return listOf(UCContentTextSectionPM(title = simpleCardContent.title + ": " + simpleCardContent.value))
    }

    private fun mapServiceContent(
        service: PredefinedUIServiceDetails,
        switchSettings: PredefinedUISwitchSettingsUI?,
        toggleMediator: PredefinedUIToggleMediator
    ): UContentToggleEntryPM {

        val toggle = if (switchSettings != null) {
            val group = toggleMediator.getServiceGroupLegacy(service.id, switchSettings)
            UCTogglePM(switchSettings, group = group)
        } else {
            null
        }

        return UContentToggleEntryPM(id = service.id, name = service.name, toggle = toggle)
    }

    private fun mapSimpleCardContent(simpleCardContent: PredefinedUISimpleCardContent): List<UCContentSectionPM> {
        val textSectionList = mutableListOf(UCContentTextSectionPM(description = simpleCardContent.description))

        if (simpleCardContent.value.isNotBlank()) {
            textSectionList.add(UCContentTextSectionPM(title = simpleCardContent.title, description = simpleCardContent.value))
        }
        return textSectionList
    }
}
