package com.usercentrics.sdk.ui.popup

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.graphics.Typeface
import android.graphics.drawable.*
import android.view.*
import android.widget.LinearLayout
import android.widget.PopupWindow
import android.widget.ScrollView
import android.widget.TextView
import androidx.core.content.ContextCompat
import androidx.core.widget.PopupWindowCompat
import com.usercentrics.sdk.models.settings.PredefinedUILanguage
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class LanguagePopup(
    private val context: Context,
    private val theme: UCThemeData
) : PopupWindow.OnDismissListener {

    private val popupWindow = PopupWindow()
    private var languageSelectedListener: ((String) -> Unit)? = null

    private val smallPadding
        get() = 8.dpToPx(context)

    private val regularPadding
        get() = 16.dpToPx(context)

    fun show(anchor: View, languages: List<PredefinedUILanguage>, selectedIsoCode: String) {
        setupWindow(languages, selectedIsoCode)

        PopupWindowCompat.showAsDropDown(popupWindow, anchor, anchor.width, 0, Gravity.NO_GRAVITY)
    }

    fun setLanguageSelectedListener(listener: (String) -> Unit): LanguagePopup {
        this.languageSelectedListener = listener
        return this
    }

    override fun onDismiss() {
        handleDismiss()
    }

    @SuppressLint("InflateParams")
    private fun setupWindow(languages: List<PredefinedUILanguage>, selectedIsoCode: String) {
        popupWindow.apply {
            contentView = buildContentView(languages, selectedIsoCode)

            width = 200.dpToPx(context)
            height = ViewGroup.LayoutParams.WRAP_CONTENT

            inputMethodMode = PopupWindow.INPUT_METHOD_FROM_FOCUSABLE
            softInputMode = WindowManager.LayoutParams.SOFT_INPUT_STATE_UNCHANGED

            isFocusable = true
            isOutsideTouchable = true

            setOnDismissListener(this@LanguagePopup)

            elevation = 12f.dpToPx(context)
            setBackgroundDrawable(containerBackground())
        }
    }

    private fun buildContentView(languages: List<PredefinedUILanguage>, selectedIsoCode: String): View {
        return ScrollView(context).apply {
            val container = LinearLayout(context).apply {
                id = R.id.ucLanguagesAvailable
                orientation = LinearLayout.VERTICAL
                setPaddingRelative(smallPadding, regularPadding, smallPadding, regularPadding)

                addLanguages(this, languages, selectedIsoCode)
            }
            addView(container)
        }
    }

    private fun containerBackground(): Drawable {
        return GradientDrawable().apply {
            cornerRadius = 4f.dpToPx(context)

            setColor(theme.colorPalette.layerBackgroundColor ?: Color.WHITE)
            setStroke(1.dpToPx(context), theme.colorPalette.tabsBorderColor)
        }
    }

    private fun addLanguages(container: ViewGroup, languages: List<PredefinedUILanguage>, selectedIsoCode: String) {
        val iterator = languages.iterator()

        while (true) {
            val language = iterator.next()

            val hasNext = iterator.hasNext()
            val bottomPadding = if (hasNext) regularPadding else smallPadding
            val isSelected = selectedIsoCode == language.isoCode

            val textView = languageTextView(language, bottomPadding, isSelected)
            container.addView(textView)

            if (!hasNext) {
                break
            }
        }
    }

    private fun languageTextView(language: PredefinedUILanguage, bottomPadding: Int, isSelected: Boolean): TextView {
        return UCTextView(context).apply {
            textSize = 16f
            text = language.fullName
            tag = language.isoCode

            setPaddingRelative(smallPadding, smallPadding, smallPadding, bottomPadding)
            setTextColor(theme.colorPalette.text100 ?: ContextCompat.getColor(context, R.color.ucDarkGray))
            setOnClickListener(::languageTextViewClickListener)

            val style = if (isSelected) Typeface.BOLD else Typeface.NORMAL
            setTypeface(theme.fonts.font, style)
        }
    }

    private fun languageTextViewClickListener(view: View) {
        dismiss()

        val languageIsoCode = view.tag as? String ?: return
        this.languageSelectedListener?.invoke(languageIsoCode)
    }

    private fun dismiss() {
        popupWindow.dismiss()
    }

    private fun handleDismiss() {
        if (popupWindow.isShowing) {
            dismiss()
        }
    }
}
