package com.usercentrics.sdk.ui.secondLayer

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.appcompat.widget.LinearLayoutCompat
import androidx.appcompat.widget.Toolbar
import androidx.core.view.isVisible
import androidx.viewpager.widget.ViewPager
import com.google.android.material.appbar.AppBarLayout
import com.google.android.material.tabs.TabLayout
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCButton
import com.usercentrics.sdk.ui.components.cards.UCCardPM
import com.usercentrics.sdk.ui.components.cards.UCControllerIdPM
import com.usercentrics.sdk.ui.secondLayer.component.adapters.PAGE_TAG
import com.usercentrics.sdk.ui.secondLayer.component.adapters.UCSecondLayerTabsPagerAdapter
import com.usercentrics.sdk.ui.secondLayer.component.footer.UCSecondLayerFooter
import com.usercentrics.sdk.ui.secondLayer.component.header.UCSecondLayerHeader
import com.usercentrics.sdk.ui.theme.UCThemeData

// Legacy UCLayer
@SuppressLint("ViewConstructor")
internal class UCSecondLayerView(
    context: Context,
    private val theme: UCThemeData
) : LinearLayoutCompat(context, null, 0) {

    private val ucFooter by lazy { findViewById<UCSecondLayerFooter>(R.id.ucFooter) }
    private val ucHeader by lazy { findViewById<UCSecondLayerHeader>(R.id.ucHeader) }
    private val ucToolbar by lazy { findViewById<Toolbar>(R.id.ucToolbar) }
    private val ucContentViewPager by lazy { findViewById<ViewPager>(R.id.ucContentViewPager) }
    private val ucAppBar by lazy { findViewById<AppBarLayout>(R.id.ucAppBar) }
    private val pagerAdapter: UCSecondLayerTabsPagerAdapter = UCSecondLayerTabsPagerAdapter(
        theme,
        ::navigateToTab,
        ::collapseHeader
    )

    private var lastSelectedTabIndex: Int? = null

    init {
        setupView()
    }

    fun bindViewModel(viewModel: UCSecondLayerViewModel) {
        viewModel.bind { content, header, footer ->
            ucHeader.bind(theme, header)
            ucFooter.bind(footer)
            bindContent(content)
        }
    }

    private fun setupView() {
        LayoutInflater.from(context).inflate(R.layout.uc_layer, this)
        orientation = VERTICAL
        setBackgroundColor(Color.WHITE)
        ucContentViewPager.adapter = pagerAdapter
        ucContentViewPager.addOnPageChangeListener(TabChangeListener())
        ucHeader.style(theme)
        ucFooter.style(theme)

        post {
            ucAppBar.bringToFront()
            ucAppBar.setExpanded(true, true)
            ucHeader.requestFocus()
            addOnGlobalFocusChangeListener()
        }
    }

    private fun bindContent(content: UCLayerContentPM) {
        pagerAdapter.contentTabs = content.tabs
        val showTabs = content.tabs.size > 1
        ucHeader.bindTabs(theme, ucContentViewPager, content.tabs.map { it.title }, showTabs)
        ucToolbar.layoutParams = ucToolbar.layoutParams.apply {
            height = if (showTabs) resources.getDimension(R.dimen.ucTabLayoutHeight).toInt() else 0
        }
        val tabIndex = lastSelectedTabIndex ?: content.initialTabIndex
        if (tabIndex > 0 && tabIndex < content.tabs.size) {
            ucContentViewPager.setCurrentItem(tabIndex, false)
        }
    }

    private fun navigateToTab(position: Int) {
        ucContentViewPager.currentItem = position
    }

    private fun collapseHeader() {
        ucAppBar.setExpanded(false, true)
    }

    private inner class TabChangeListener : ViewPager.OnPageChangeListener {
        override fun onPageScrolled(p1: Int, p2: Float, p3: Int) {}
        override fun onPageScrollStateChanged(p1: Int) {}
        override fun onPageSelected(position: Int) {
            ucContentViewPager.findViewWithTag<View>(PAGE_TAG + position)?.requestFocus()
            lastSelectedTabIndex = position
        }
    }

    private fun addOnGlobalFocusChangeListener() {
        viewTreeObserver.addOnGlobalFocusChangeListener { oldFocus, newFocus ->
            if (oldFocus == null || newFocus == null) return@addOnGlobalFocusChangeListener
            when {
                oldFocus.id == R.id.ucHeader && newFocus.id == R.id.ucHeader -> {
                    val page = ucContentViewPager.findViewWithTag<View>(PAGE_TAG + ucContentViewPager.currentItem)
                    page?.findViewById<View>(R.id.ucCardTitle)?.run { requestFocus() }
                }

                oldFocus is UCButton && newFocus.id == R.id.ucHeader -> {
                    if (ucHeader.findViewById<View>(R.id.ucHeaderLanguageIcon).isVisible)
                        ucHeader.findViewById<View>(R.id.ucHeaderLanguageIcon)?.run { requestFocus() }
                    else {
                        ucHeader.findViewById<View>(R.id.ucHeaderCloseButton)?.run { requestFocus() }
                    }
                    ucAppBar.setExpanded(true, true)
                }

                newFocus.id == R.id.ucHeader -> requestFocusSelectedTabView()
            }
        }
    }

    private fun requestFocusSelectedTabView() {
        val tabLayout = ucHeader.findViewById<TabLayout>(R.id.ucHeaderTabLayout)
        (tabLayout.getChildAt(0) as? ViewGroup)?.getChildAt(ucContentViewPager.currentItem)?.run {
            isFocusable = true
            isFocusableInTouchMode = true
            requestFocus()
            ucAppBar.setExpanded(true, true)
        }
    }
}

internal class UCLayerContentPM(
    val initialTabIndex: Int,
    val tabs: List<UCLayerTabPM>
)

internal class UCLayerTabPM(val title: String, val content: List<UCCardsContentPM>)

internal sealed class UCLayerTabContentPM

internal class UCCardsContentPM(
    val title: String?,
    val cards: List<UCCardPM>,
    val controllerId: UCControllerIdPM?
) : UCLayerTabContentPM()
