package com.usercentrics.sdk.ui.secondLayer.component.adapters

import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.RecyclerView
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCControllerId
import com.usercentrics.sdk.ui.components.UCSectionTitle
import com.usercentrics.sdk.ui.components.cards.*
import com.usercentrics.sdk.ui.extensions.set
import com.usercentrics.sdk.ui.extensions.setRVMargins
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class UCSecondLayerCardsAdapter(
    private val theme: UCThemeData,
    private val onMoreInfo: ((String) -> Unit)?,
    private val centerCardBy: (cardY: Int) -> Unit,
) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {

    companion object {
        private const val CONTROLLER_ID_TYPE = 841
        private const val SECTION_TITLE_TYPE = 842
        private const val CARD_TYPE = 843
    }

    var cardComponents: List<UCCardComponent> = emptyList()
        set(value) {
            field = value
            notifyDataSetChanged()
        }

    private val expandedPositions: MutableSet<Int> = mutableSetOf()

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
        return when (viewType) {
            SECTION_TITLE_TYPE -> UCSectionTitleViewHolder(theme, UCSectionTitle(parent.context))
            CARD_TYPE -> UCCardViewHolder(theme, UCCard(parent.context))
            CONTROLLER_ID_TYPE -> UCControllerIdViewHolder(theme, UCControllerId(parent.context))
            else -> throw RuntimeException("not implemented")
        }
    }

    override fun getItemViewType(position: Int): Int {
        return when (cardComponents[position]) {
            is UCSectionTitlePM -> SECTION_TITLE_TYPE
            is UCCardPM -> CARD_TYPE
            is UCControllerIdPM -> CONTROLLER_ID_TYPE
        }
    }

    override fun getItemCount(): Int {
        return cardComponents.size
    }

    private fun isLastItem(position: Int) = position == itemCount - 1

    override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
        val cardComponent = cardComponents[position]
        when (holder) {
            is UCSectionTitleViewHolder -> {
                holder.bind(cardComponent as UCSectionTitlePM)
            }
            is UCCardViewHolder -> {
                holder.bindCard(
                    cardPM = cardComponent as UCCardPM,
                    onMoreInfo = onMoreInfo,
                    isExpanded = expandedPositions.contains(position),
                    isLastItem = isLastItem(position)
                ) { isExpanded ->
                    expandedPositions.set(position, isExpanded)
                    if (isExpanded) {
                        val cardLocation = intArrayOf(0, 0)
                        holder.itemView.getLocationOnScreen(cardLocation)
                        val cardY = cardLocation[1]
                        centerCardBy(cardY)
                    }
                    notifyItemChanged(position)
                }
            }
            is UCControllerIdViewHolder -> {
                holder.bind(cardComponent as UCControllerIdPM)
            }
        }
    }

    fun cardPosition(cardId: String): Int {
        return cardComponents.indexOfFirst { card -> (card as? UCCardPM)?.id == cardId }
    }

    fun setExpanded(position: Int, shouldNotify: Boolean = true) {
        val changed = expandedPositions.set(position, true)
        if (changed && shouldNotify) {
            notifyItemChanged(position)
        }
    }

    fun collapseAll(shouldNotify: Boolean = true) {
        val removedPositions = expandedPositions.toHashSet()
        expandedPositions.clear()
        if (shouldNotify) {
            removedPositions.forEach {
                notifyItemChanged(it)
            }
        }
    }
}

internal class UCCardViewHolder(
    private val theme: UCThemeData,
    itemView: View
) : RecyclerView.ViewHolder(itemView) {

    private val card = itemView as UCCard

    init {
        card.style(theme)
    }

    fun bindCard(cardPM: UCCardPM, onMoreInfo: ((String) -> Unit)?, isExpanded: Boolean, isLastItem: Boolean, isExpandedListener: (Boolean) -> Unit) {
        card.bindCard(theme, cardPM, isExpanded, isExpandedListener, onMoreInfo)
        card.setRVMargins(
            horizontalMarginPx = itemView.resources.getDimension(R.dimen.ucCardHorizontalMargin).toInt(),
            verticalMarginPx = itemView.resources.getDimension(R.dimen.ucCardVerticalMargin).toInt(),
            hasBottom = isLastItem
        )
    }
}

internal class UCControllerIdViewHolder(theme: UCThemeData, itemView: View) : RecyclerView.ViewHolder(itemView) {

    private val controllerId = itemView as UCControllerId

    init {
        controllerId.style(theme)
        controllerId.setRVMargins(
            horizontalMarginPx = itemView.resources.getDimension(R.dimen.ucControllerIdHorizontalMargin)
                .toInt(),
            verticalMarginPx = itemView.resources.getDimension(R.dimen.ucControllerIdVerticalMargin)
                .toInt(),
            hasBottom = true
        )
    }

    fun bind(model: UCControllerIdPM) {
        controllerId.bind(model)
        controllerId.setRVMargins(
            horizontalMarginPx = itemView.resources.getDimension(R.dimen.ucCardHorizontalMargin)
                .toInt(),
            verticalMarginPx = itemView.resources.getDimension(R.dimen.ucCardVerticalMargin)
                .toInt(),
            hasBottom = true
        )
    }
}

internal class UCSectionTitleViewHolder(theme: UCThemeData, itemView: View) : RecyclerView.ViewHolder(itemView) {

    private val sectionTitle = itemView as UCSectionTitle

    init {
        sectionTitle.style(theme)
    }

    fun bind(model: UCSectionTitlePM) {
        sectionTitle.bind(model)
    }
}
