package com.usercentrics.sdk.ui.secondLayer.component.footer

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.LinearLayout
import androidx.annotation.AttrRes
import androidx.constraintlayout.widget.ConstraintLayout
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.accessibility.accessibleTouchTarget
import com.usercentrics.sdk.ui.components.UCButton
import com.usercentrics.sdk.ui.components.UCButtonSettings
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.components.UCToggle
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class UCSecondLayerFooter : ConstraintLayout {

    private val ucFooterSwitch by lazy { findViewById<UCToggle>(R.id.ucFooterSwitch) }
    private val ucFooterSwitchText by lazy { findViewById<UCTextView>(R.id.ucFooterSwitchText) }
    private val ucFooterButtonsContainer by lazy { findViewById<LinearLayout>(R.id.ucFooterButtonsContainer) }
    private val ucFooterTextProvider by lazy { findViewById<UCTextView>(R.id.ucFooterTextProvider) }
    private val ucFooterDivider by lazy { findViewById<View>(R.id.ucFooterDivider) }

    private lateinit var viewModel: UCSecondLayerFooterViewModel

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initLayout(context)
    }

    private fun initLayout(context: Context) {
        LayoutInflater.from(context).inflate(R.layout.uc_footer, this)
    }

    fun style(theme: UCThemeData) {
        ucFooterSwitch.styleToggle(theme)
        ucFooterSwitchText.styleBody(theme)
        ucFooterTextProvider.styleTiny(theme)
        ucFooterDivider.setBackgroundColor(theme.colorPalette.tabsBorderColor)
        theme.colorPalette.layerBackgroundColor?.let { setBackgroundColor(it) }
    }

    fun bind(model: UCSecondLayerFooterViewModel) {
        this.viewModel = model
        bindSwitch()
        bindProviderLink()
        bindButtons()
        invalidate()
    }

    private fun bindSwitch() {
        val switchLabel = viewModel.optOutToggle
        if (switchLabel?.isNotBlank() == true) {
            ucFooterSwitch.visibility = View.VISIBLE
            ucFooterSwitchText.visibility = View.VISIBLE
            ucFooterSwitchText.text = switchLabel
            ucFooterSwitch.currentState = viewModel.optOutToggleInitialValue
            ucFooterSwitch.setListener {
                viewModel.onOptOutSwitchChanged(it)
            }
            ucFooterSwitchText.setOnClickListener { ucFooterSwitch.toggle() }
            ucFooterSwitchText.accessibleTouchTarget()
        } else {
            ucFooterSwitch.visibility = View.GONE
            ucFooterSwitchText.visibility = View.GONE
        }
    }

    private fun bindButtons() {
        ucFooterButtonsContainer.removeAllViews()
        val buttons = viewModel.buttons
        buttons.forEachIndexed { index, row ->
            val rowButtons = bindRowButtons(row, index == buttons.lastIndex)

            if (rowButtons.size == 1) {
                createSingleButtonRow(rowButtons[0])
            } else {
                createHorizontalButtonRow(rowButtons)
            }
        }
    }

    private fun bindRowButtons(row: List<UCButtonSettings>, isLastRow: Boolean): List<UCButton> {
        return row.mapIndexed { index, settings ->
            val button = UCButton(context)

            button.layoutParams = LinearLayout.LayoutParams(
                LinearLayout.LayoutParams.MATCH_PARENT,
                LinearLayout.LayoutParams.MATCH_PARENT
            ).apply {
                this.weight = 1f
                val isLastButtonOfTheRow = index == row.lastIndex

                marginStart = 0
                topMargin = 0
                marginEnd = if (!isLastButtonOfTheRow) 8.dpToPx(context) else 0
                bottomMargin = if (!isLastRow) 8.dpToPx(context) else 0
            }
            button.setup(settings) { viewModel.onButtonClick(settings.type) }

            button
        }
    }

    private fun createSingleButtonRow(button: UCButton) {
        ucFooterButtonsContainer.addView(button)
    }

    private fun createHorizontalButtonRow(buttons: List<UCButton>) {
        val horizontalRow = LinearLayout(context)
        horizontalRow.layoutParams = LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.WRAP_CONTENT
        )
        horizontalRow.orientation = LinearLayout.HORIZONTAL

        for (buttonView in buttons) {
            horizontalRow.addView(buttonView)
        }
        ucFooterButtonsContainer.addView(horizontalRow)
    }

    private fun bindProviderLink() {
        val poweredByLabel = viewModel.poweredBy
        val hasProvider = if (poweredByLabel != null) {
            ucFooterTextProvider.visibility = View.VISIBLE
            ucFooterTextProvider.text = poweredByLabel
            true
        } else {
            ucFooterTextProvider.visibility = View.GONE
            false
        }
        ucFooterButtonsContainer.layoutParams = ucFooterButtonsContainer.layoutParams.apply {
            this as LayoutParams

            marginStart = marginStart
            marginEnd = marginEnd
            topMargin = topMargin
            bottomMargin = if (hasProvider) 8.dpToPx(context) else 16.dpToPx(context)
        }
    }
}
