package com.usercentrics.sdk.unity

import android.app.Activity
import android.content.Context
import android.os.Handler
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import com.usercentrics.sdk.*
import com.usercentrics.sdk.unity.data.UnityPredefinedUIResponse
import com.usercentrics.sdk.unity.data.UnityServiceConsent
import com.usercentrics.sdk.unity.model.UnityBannerSettings
import com.usercentrics.sdk.unity.model.UnityFirstLayerStyleSettings

@Suppress("unused")
class UsercentricsUnityBanner(private val context: Context) {

    private var statusBarWasShown = false
    private val jsonParser = UnityUIJsonParser()

    fun showFirstLayer(bannerSettings: String? = null) {
        val unityBannerSettings = if (bannerSettings != null) {
            jsonParser.decodeFromString(bannerSettings)
        } else {
            UnityBannerSettings()
        }

        runOnMainThread {
            val generalStyleSettings = unityBannerSettings.generalStyleSettings
            showStatusBarIfNeeded(statusBarColor = generalStyleSettings?.statusBarColor, windowFullscreen = generalStyleSettings?.windowFullscreen)

            UsercentricsBanner(context, unityBannerSettings.toBannerSettings()).showFirstLayer(::sendHandleBannerResponseUnityMessage)
        }
    }

    private fun parseFirstLayerStyleSettings(styleSettingsJson: String): FirstLayerStyleSettings {
        val settings = jsonParser.decodeFromString<UnityFirstLayerStyleSettings>(styleSettingsJson)
        return settings.toFirstLayerSettings()
    }

    fun showSecondLayer(bannerSettings: String? = null) {
        val unityBannerSettings = if (bannerSettings != null) {
            jsonParser.decodeFromString(bannerSettings)
        } else {
            UnityBannerSettings()
        }

        runOnMainThread {
            val generalStyleSettings = unityBannerSettings.generalStyleSettings
            showStatusBarIfNeeded(generalStyleSettings?.statusBarColor, generalStyleSettings?.windowFullscreen)

            UsercentricsBanner(context, unityBannerSettings.toBannerSettings()).showSecondLayer(::sendHandleBannerResponseUnityMessage)
        }
    }

    private fun sendHandleBannerResponseUnityMessage(consentUserResponse: UsercentricsConsentUserResponse?) {
        runOnMainThread { restoreStateOfStatusBar() }

        UsercentricsUnity.handleBannerResponse(consentUserResponse.toUnityPredefinedUIResponse())
    }

    private fun runOnMainThread(block: () -> Unit) {
        val handler = Handler(context.mainLooper)
        handler.post(block)
    }

    private fun showStatusBarIfNeeded(statusBarColor: String?, windowFullscreen: Boolean?) {
        if (statusBarColor == null || windowFullscreen == true) {
            return
        }

        val window = (context as? Activity)?.window ?: return

        val insetsController = WindowCompat.getInsetsController(window, window.decorView)
        insetsController.show(WindowInsetsCompat.Type.statusBars())

        statusBarWasShown = true
    }

    private fun restoreStateOfStatusBar() {
        if (!statusBarWasShown) {
            return
        }

        val window = (context as? Activity)?.window ?: return

        val insetsController = WindowCompat.getInsetsController(window, window.decorView)
        insetsController.hide(WindowInsetsCompat.Type.statusBars())

        statusBarWasShown = false
    }
}

internal fun UsercentricsConsentUserResponse?.toUnityPredefinedUIResponse(): UnityPredefinedUIResponse? {
    if (this == null) {
        return null
    }

    return UnityPredefinedUIResponse(
        userInteraction = userInteraction.toPredefinedUIInteraction(),
        consents = this.consents.map { UnityServiceConsent(it) },
        controllerId = this.controllerId
    )
}
