package com.usercentrics.sdk.unity.model

import com.usercentrics.sdk.*
import com.usercentrics.sdk.ui.extensions.unsafeParseColor
import com.usercentrics.sdk.ui.extensions.withAlpha
import kotlinx.serialization.Contextual
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

@Serializable
internal data class UnityBannerSettings(
    val generalStyleSettings: UnityGeneralStyleSettings? = null,
    val firstLayerStyleSettings: UnityFirstLayerStyleSettings? = null,
    val secondLayerStyleSettings: UnitySecondLayerStyleSettings? = null,
    val variantName: String? = null,
) {

    fun toBannerSettings(): BannerSettings {
        return BannerSettings(
            generalStyleSettings = this.generalStyleSettings?.toGeneralStyleSettings(),
            firstLayerStyleSettings = this.firstLayerStyleSettings?.toFirstLayerSettings(),
            secondLayerStyleSettings = this.secondLayerStyleSettings?.toSecondLayerSettings(),
            variantName = variantName,
        )
    }
}

@Serializable
internal data class UnityGeneralStyleSettings(
    @SerialName("androidDisableSystemBackButton")
    val disableSystemBackButton: Boolean? = null,

    @SerialName("androidStatusBarColor")
    val statusBarColor: String? = null,

    @SerialName("androidWindowFullscreen")
    val windowFullscreen: Boolean? = null,

    val textColor: String? = null,
    val layerBackgroundColor: String? = null,
    val layerBackgroundSecondaryColor: String? = null,
    val linkColor: String? = null,
    val tabColor: String? = null,
    val bordersColor: String? = null,
    val toggleStyleSettings: UnityToggleStyleSettings? = null,
    val logoImageUrl: String? = null,
    @Contextual val links: UnityLegalLinksSettings = UnityLegalLinksSettings.UNDEFINED,
) {

    fun toGeneralStyleSettings(): GeneralStyleSettings {
        return GeneralStyleSettings(
            disableSystemBackButton = this.disableSystemBackButton,
            statusBarColor = parseColor(this.statusBarColor),
            windowFullscreen = this.windowFullscreen,
            textColor = parseColor(this.textColor),
            layerBackgroundColor = parseColor(this.layerBackgroundColor),
            layerBackgroundSecondaryColor = parseColor(this.layerBackgroundSecondaryColor),
            linkColor = parseColor(this.linkColor),
            tabColor = parseColor(this.tabColor),
            bordersColor = parseColor(this.bordersColor),
            toggleStyleSettings = this.toggleStyleSettings?.toToggleStyleSettings(),
            logo = if (logoImageUrl.isNullOrBlank()) {
                null
            } else {
                UsercentricsImage.ImageUrl(this.logoImageUrl)
            },
            links = this.links.toLegalLinksSettings(),
        )
    }
}

@Serializable
internal data class UnityFirstLayerStyleSettings(
    @Contextual val layout: UnityUsercentricsLayout? = null,
    val headerImage: UnityHeaderImageSettings? = null,
    val title: UnityTitleSettings? = null,
    val message: UnityMessageSettings? = null,
    val buttonLayout: UnityButtonLayout? = null,
    val backgroundColor: String? = null,
    val cornerRadius: Float? = null,
    val overlayColor: String? = null,
    val overlayAlpha: Float = 1f,
) {

    fun toFirstLayerSettings(): FirstLayerStyleSettings {
        return FirstLayerStyleSettings(
            layout = layout?.toUsercentricsLayout(),
            headerImage = headerImage?.toHeaderImageSettings(),
            title = title?.toTitleSettings(),
            message = message?.toMessageSettings(),
            buttonLayout = buttonLayout?.toButtonLayout(),
            backgroundColor = parseColor(backgroundColor),
            cornerRadius = optionalValueOrNull(cornerRadius)?.toInt(),
            overlayColor = parseColor(overlayColor?.withAlpha(overlayAlpha.toDouble())),
        )
    }
}

@Serializable
internal data class UnityHeaderImageSettings(
    @Contextual val imageType: UnityHeaderImageType = UnityHeaderImageType.UNDEFINED,
    val imageUrl: String? = null,
    @Contextual val alignment: UnitySectionAlignment = UnitySectionAlignment.UNDEFINED,
    val height: Float? = null,
) {

    fun toHeaderImageSettings(): HeaderImageSettings? {
        if (imageUrl.isNullOrBlank()) {
            return null
        }

        return when (imageType) {
            UnityHeaderImageType.UNDEFINED -> {
                null
            }

            UnityHeaderImageType.EXTENDED -> {
                HeaderImageSettings.ExtendedLogoSettings(image = UsercentricsImage.ImageUrl(imageUrl))
            }

            UnityHeaderImageType.CUSTOM -> {
                HeaderImageSettings.LogoSettings(
                    image = UsercentricsImage.ImageUrl(imageUrl),
                    alignment = alignment.toSectionAlignment(),
                    heightInDp = optionalValueOrNull(height)
                )
            }

            UnityHeaderImageType.HIDDEN -> {
                HeaderImageSettings.Hidden
            }

        }
    }
}

@Serializable
internal data class UnityTitleSettings(
    @Contextual val alignment: UnitySectionAlignment = UnitySectionAlignment.UNDEFINED,
    val textSize: Float? = null,
    val textColor: String? = null,
) {
    fun toTitleSettings(): TitleSettings {
        return TitleSettings(
            textSizeInSp = optionalValueOrNull(textSize),
            alignment = alignment.toSectionAlignment(),
            textColor = parseColor(textColor),
        )
    }
}

@Serializable
internal data class UnityMessageSettings(
    val textSize: Float? = null,
    @Contextual val alignment: UnitySectionAlignment = UnitySectionAlignment.UNDEFINED,
    val textColor: String? = null,
    val linkTextColor: String? = null,
    val underlineLink: Boolean? = null,
) {
    fun toMessageSettings(): MessageSettings {
        return MessageSettings(
            textSizeInSp = optionalValueOrNull(textSize),
            alignment = alignment.toSectionAlignment(),
            textColor = parseColor(textColor),
            linkTextColor = parseColor(linkTextColor),
            underlineLink = underlineLink,
        )
    }
}

@Serializable
internal data class UnityButtonLayout(
    @Contextual val type: UnityButtonLayoutType = UnityButtonLayoutType.UNDEFINED,
    val buttons: List<UnityButtonSettings> = emptyList(),
    val gridButtons: List<UnityButtonSettingsRow> = emptyList(),
) {
    fun toButtonLayout(): ButtonLayout? {
        return when (type) {
            UnityButtonLayoutType.UNDEFINED -> {
                null
            }

            UnityButtonLayoutType.COLUMN -> {
                ButtonLayout.Column(buttons = buttons.mapNotNull { it.toButtonSettings() })
            }

            UnityButtonLayoutType.ROW -> {
                ButtonLayout.Row(buttons = buttons.mapNotNull { it.toButtonSettings() })
            }

            UnityButtonLayoutType.GRID -> {
                ButtonLayout.Grid(buttons = gridButtons.map { buttonSettingsRow -> buttonSettingsRow.buttons.mapNotNull { it.toButtonSettings() } })
            }
        }
    }
}

@Serializable
internal data class UnityButtonSettings(
    @Contextual val type: UnityButtonType? = null,
    val textSize: Float? = null,
    val textColor: String? = null,
    val backgroundColor: String? = null,
    val cornerRadius: Float? = null,
    val isAllCaps: Boolean? = null,
) {

    fun toButtonSettings(): ButtonSettings? {
        if (type == null) {
            return null
        }

        return ButtonSettings(
            type = type.toButtonType(),
            textSizeInSp = optionalValueOrNull(textSize),
            textColor = parseColor(textColor),
            backgroundColor = parseColor(backgroundColor),
            cornerRadius = optionalValueOrNull(cornerRadius)?.toInt(),
            isAllCaps = isAllCaps,
        )
    }
}

@Serializable
internal data class UnityButtonSettingsRow(
    val buttons: List<UnityButtonSettings> = emptyList()
)

@Serializable
internal data class UnitySecondLayerStyleSettings(
    val showCloseButton: Boolean? = null,
) {
    fun toSecondLayerSettings(): SecondLayerStyleSettings {
        return SecondLayerStyleSettings(showCloseButton = showCloseButton)
    }
}

@Serializable
internal data class UnityToggleStyleSettings(
    val activeBackgroundColor: String? = null,
    val inactiveBackgroundColor: String? = null,
    val disabledBackgroundColor: String? = null,
    val activeThumbColor: String? = null,
    val inactiveThumbColor: String? = null,
    val disabledThumbColor: String? = null,
) {
    fun toToggleStyleSettings(): ToggleStyleSettings {
        return ToggleStyleSettings(
            activeBackgroundColor = parseColor(this.activeBackgroundColor),
            inactiveBackgroundColor = parseColor(this.inactiveBackgroundColor),
            disabledBackgroundColor = parseColor(this.disabledBackgroundColor),
            activeThumbColor = parseColor(this.activeThumbColor),
            inactiveThumbColor = parseColor(this.inactiveThumbColor),
            disabledThumbColor = parseColor(this.disabledThumbColor),
        )
    }
}

@Serializable
internal enum class UnityHeaderImageType {

    UNDEFINED,
    EXTENDED,
    CUSTOM,
    HIDDEN,
}

@Serializable
internal enum class UnitySectionAlignment {

    UNDEFINED,
    START,
    CENTER,
    END;

    fun toSectionAlignment(): SectionAlignment? {
        return when (this) {
            UNDEFINED -> null
            START -> SectionAlignment.START
            CENTER -> SectionAlignment.CENTER
            END -> SectionAlignment.END
        }
    }
}

@Serializable
internal enum class UnityUsercentricsLayout {

    UNDEFINED,
    FULL,
    SHEET,
    POPUP_BOTTOM,
    POPUP_CENTER;

    fun toUsercentricsLayout(): UsercentricsLayout? {
        return when (this) {
            UNDEFINED -> null
            FULL -> UsercentricsLayout.Full
            SHEET -> UsercentricsLayout.Sheet
            POPUP_CENTER -> UsercentricsLayout.Popup(PopupPosition.CENTER)
            POPUP_BOTTOM -> UsercentricsLayout.Popup(PopupPosition.BOTTOM)
        }
    }
}

@Serializable
internal enum class UnityButtonLayoutType {

    UNDEFINED,
    COLUMN,
    ROW,
    GRID
}

@Serializable
internal enum class UnityButtonType {

    ACCEPT_ALL,
    DENY_ALL,
    MORE,
    SAVE;

    fun toButtonType(): ButtonType {
        return when (this) {
            ACCEPT_ALL -> ButtonType.ACCEPT_ALL
            DENY_ALL -> ButtonType.DENY_ALL
            MORE -> ButtonType.MORE
            SAVE -> ButtonType.SAVE
        }
    }
}

@Serializable
internal enum class UnityLegalLinksSettings {

    UNDEFINED,
    FIRST_LAYER_ONLY,
    SECOND_LAYER_ONLY,
    BOTH,
    HIDDEN;

    fun toLegalLinksSettings(): LegalLinksSettings? {
        return when (this) {
            UNDEFINED -> null
            FIRST_LAYER_ONLY -> LegalLinksSettings.FIRST_LAYER_ONLY
            SECOND_LAYER_ONLY -> LegalLinksSettings.SECOND_LAYER_ONLY
            BOTH -> LegalLinksSettings.BOTH
            HIDDEN -> LegalLinksSettings.HIDDEN
        }
    }
}

private fun <T> optionalValueOrNull(value: T?): T? {
    if (value == null) {
        return null
    }
    when (value) {
        is Float -> {
            if (value <= 0.0f) {
                return null
            }
            return value
        }

        is Int -> {
            if (value <= 0) {
                return null
            }
            return value
        }

        else -> {
            throw IllegalArgumentException("[USERCENTRICS] error when parsing optional value $value - reason: unknown type")
        }
    }
}

private fun parseColor(value: String?): Int? {
    if (value.isNullOrBlank()) {
        return null
    }
    try {
        return value.unsafeParseColor()
    } catch (e: Exception) {
        throw IllegalArgumentException("[USERCENTRICS] error when parsing color with HEX<${value}>", e)
    }
}
