package com.usercentrics.sdk

import android.content.Context
import androidx.annotation.ColorInt
import com.usercentrics.sdk.analytics.UsercentricsAnalyticsManagerImpl
import com.usercentrics.sdk.logger.UsercentricsUILoggerImpl
import com.usercentrics.sdk.models.settings.PredefinedUIAriaLabels
import com.usercentrics.sdk.models.settings.PredefinedUIVariant
import com.usercentrics.sdk.predefinedUI.PredefinedUIApplication
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.PredefinedUIFactoryHolder
import com.usercentrics.sdk.ui.banner.*
import com.usercentrics.sdk.ui.extensions.openUrl
import com.usercentrics.sdk.ui.extensions.safeShowBanner
import com.usercentrics.sdk.ui.theme.UCThemeData
import java.lang.ref.WeakReference

class UsercentricsBanner(
    context: Context,
    private val settings: BannerSettings? = null,
) {

    private var dialog: UsercentricsDialog? = null
    private var onDismissCallback: ((UsercentricsConsentUserResponse?) -> Unit)? = null

    private val contextReference = WeakReference(context)

    private fun getContext(): Context? {
        return contextReference.get()
    }

    fun showFirstLayer(callback: (UsercentricsConsentUserResponse?) -> Unit) {
        doShowFirstLayer(settings?.firstLayerStyleSettings?.layout, callback)
    }

    private fun doShowFirstLayer(customLayout: UsercentricsLayout?, callback: (UsercentricsConsentUserResponse?) -> Unit) {
        onDismissCallback = callback

        val instance = Usercentrics.instance

        instance.getUIFactoryHolder(settings?.variantName, customLayout?.predefinedUIVariant()) { predefinedUIFactoryHolder ->

            val layout = customLayout ?: predefinedUIFactoryHolder.uiHolder.data.settings.firstLayerV2.layout.toUsercentricsLayout()

            getContext()?.let { context ->
                context.safeShowBanner {
                    getDialog(
                        context = context,
                        usercentricsSDK = instance,
                        customOverlayColor = settings?.firstLayerStyleSettings?.overlayColor,
                        slideTransitionEnabled = layout == UsercentricsLayout.Sheet,
                        predefinedUIFactoryHolder = predefinedUIFactoryHolder,
                    ).showFirstLayer(layout)
                }
            }
        }
    }

    fun showSecondLayer(callback: (UsercentricsConsentUserResponse?) -> Unit) {
        onDismissCallback = callback

        val instance = Usercentrics.instance

        instance.getUIFactoryHolder(settings?.variantName, PredefinedUIVariant.SECOND_LAYER) { predefinedUIFactoryHolder ->
            
           getContext()?.let { context ->
               context.safeShowBanner {
                   getDialog(
                       context = context,
                       usercentricsSDK = instance,
                       customOverlayColor = null,
                       slideTransitionEnabled = false,
                       predefinedUIFactoryHolder = predefinedUIFactoryHolder,
                   ).showSecondLayer(null)
               }
           }
        }
    }

    private fun getDialog(
        context: Context,
        usercentricsSDK: UsercentricsSDK,
        @ColorInt customOverlayColor: Int?,
        slideTransitionEnabled: Boolean,
        predefinedUIFactoryHolder: PredefinedUIFactoryHolder
    ): UsercentricsDialog {
        return dialog ?: run {
            val uiHolder = predefinedUIFactoryHolder.uiHolder

            initDependencyManager(
                usercentricsSDK = usercentricsSDK,
                uiApplication = predefinedUIFactoryHolder.uiApplication,
                ariaLabels = uiHolder.data.settings.internationalizationLabels.ariaLabels
            )

            val customization = uiHolder.data.settings.customization
            val theme = UCThemeData.createFrom(customization, settings?.generalStyleSettings)

            dialog = UsercentricsDialog(
                context = context,
                theme = theme,
                bannerSettings = settings,
                customOverlayColor = customOverlayColor,
                slideTransitionEnabled = slideTransitionEnabled,
                coordinator = BannerCoordinator(),
                uiHolder = uiHolder,
            )

            dialog!!
        }
    }

    fun dismiss() {
        dialog?.dismiss()
        dialog = null
        onDismissCallback = null
        tearDown()
    }

    private fun tearDown() {
        PredefinedUIDependencyManager.tearDown()
    }

    private fun initDependencyManager(usercentricsSDK: UsercentricsSDK, uiApplication: PredefinedUIApplication?, ariaLabels: PredefinedUIAriaLabels) {
        uiApplication ?: return

        PredefinedUIDependencyManager.boot(
            logger = UsercentricsUILoggerImpl(uiApplication.loggerLevel),
            cookieInformationService = uiApplication.cookieInformationService,
            analyticsManager = UsercentricsAnalyticsManagerImpl(usercentricsSDK),
            ariaLabels = ariaLabels,
        )
    }

    private inner class BannerCoordinator : UCBannerCoordinator {

        override fun navigateToUrl(url: String?) {
            getContext()?.openUrl(url ?: "")
        }

        override fun navigateToSecondLayer(initialState: SecondLayerInitialState) {
            dialog?.showSecondLayer(initialState)
        }

        override fun finishCMP(response: UsercentricsConsentUserResponse?) {
            onDismissCallback?.invoke(response)
            onDismissCallback = null
            dismiss()
        }
    }
}

