package com.usercentrics.sdk

import android.app.Activity
import android.content.Context
import android.content.res.Configuration
import android.view.View
import androidx.annotation.ColorInt
import com.usercentrics.sdk.ui.PredefinedUIHolder
import com.usercentrics.sdk.ui.extensions.themed
import com.usercentrics.sdk.ui.theme.UCThemeData
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediator
import androidx.appcompat.app.AlertDialog
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.banner.SecondLayerInitialState
import com.usercentrics.sdk.ui.banner.UCBannerContainerView
import com.usercentrics.sdk.ui.banner.UCBannerCoordinator
import com.usercentrics.sdk.ui.banner.UCBannerTransition
import com.usercentrics.sdk.ui.banner.UCBannerTransitionImpl
import com.usercentrics.sdk.ui.extensions.isActivityFinishingOrDestroyed
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerViewModelImpl
import com.usercentrics.sdk.ui.secondLayer.UCSecondLayerViewModelImpl
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediatorImpl
import java.lang.ref.WeakReference

internal class UsercentricsDialog(
    context: Context,
    private val theme: UCThemeData,
    private val bannerSettings: BannerSettings? = null,
    @ColorInt customOverlayColor: Int?,
    slideTransitionEnabled: Boolean,
    private val coordinator: UCBannerCoordinator,
    private val uiHolder: PredefinedUIHolder,
) {

    private val contextReference = WeakReference(context)
    private val activityStatusBarColor: Int? = (getContext() as? Activity)?.window?.statusBarColor

    private val themedContext: Context? = getContext()?.themed()
    private val landscapeMode: Boolean by lazy {
        isOrientationLandscape() || isTablet()
    }

    private var toggleMediator: PredefinedUIToggleMediator? = null

    private var alertDialog: AlertDialog? = null
    private var bannerContainerView: UCBannerContainerView? = null
    private var bannerTransition: UCBannerTransition? = null

    init {
        val currentContext = getContext()

        val isFullScreenMode = bannerSettings?.generalStyleSettings?.windowFullscreen ?: false

        if (currentContext != null) {
            toggleMediator = PredefinedUIToggleMediatorImpl()
            bannerContainerView = UCBannerContainerView(
                context = currentContext,
                theme = theme,
                themedContext = themedContext ?: currentContext.themed(),
                isFullScreen = isFullScreenMode
            ).apply {
                id = R.id.ucBannerContainer
                visibility = View.INVISIBLE
                bannerTransition = UCBannerTransitionImpl(currentContext, theme, customOverlayColor, this, slideTransitionEnabled)
            }

            alertDialog = PredefinedUIAlertDialogFactory.create(
                context = themedContext ?: currentContext.themed(),
                rootView = bannerTransition!!.rootView,
                cancelable = bannerSettings?.generalStyleSettings?.disableSystemBackButton?.not() ?: true,
                statusBarIsCustomized = bannerSettings?.generalStyleSettings?.statusBarColor != null,
                fullscreen = isFullScreenMode,
                onDismissCallback = {
                    val feedback = uiHolder.consentManager.close()
                    coordinator.finishCMP(feedback.toUserResponse())
                }
            )

            bannerTransition?.enter()
        }
    }

    private fun getContext(): Context? {
        return contextReference.get()
    }

    fun showFirstLayer(layout: UsercentricsLayout) {
        val viewModel = UCFirstLayerViewModelImpl(
            layout = layout,
            layerSettings = uiHolder.data.settings.firstLayerV2,
            consentManager = uiHolder.consentManager,
            buttonLabels = uiHolder.data.settings.internationalizationLabels.firstLayerButtonLabels,
            theme = theme,
            generalLogo = bannerSettings?.generalStyleSettings?.logo,
            settings = bannerSettings?.firstLayerStyleSettings,
            coordinator = coordinator,
            toggleMediator = toggleMediator!!,
            landscapeMode = landscapeMode,
            linksSettings = linksSettings,
            ariaLabels = uiHolder.data.settings.internationalizationLabels.ariaLabels,
            statusBarColor = bannerSettings?.generalStyleSettings?.statusBarColor
        )

        bannerContainerView?.showFirstLayer(
            viewModel = viewModel,
            layout = layout,
            customBackgroundColor = bannerSettings?.firstLayerStyleSettings?.backgroundColor,
            customCornerRadiusInDp = bannerSettings?.firstLayerStyleSettings?.cornerRadius,
        )
    }

    private val linksSettings: LegalLinksSettings by lazy { bannerSettings?.generalStyleSettings?.links ?: LegalLinksSettings.BOTH }

    private fun isOrientationLandscape(): Boolean {
        return getContext()?.resources?.configuration?.orientation == Configuration.ORIENTATION_LANDSCAPE
    }

    private fun isTablet(): Boolean {
        return getContext()?.let { context ->
            (context.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK) >= Configuration.SCREENLAYOUT_SIZE_LARGE
        } ?: false
    }

    fun showSecondLayer(initialState: SecondLayerInitialState?) {
        val context = getContext() ?: return

        val viewModel = UCSecondLayerViewModelImpl(
            context = context,
            consentManager = uiHolder.consentManager,
            viewHandlers = uiHolder.viewHandlers,
            layerSettings = uiHolder.data.settings.secondLayerV2,
            controllerId = uiHolder.data.controllerId,
            settings = bannerSettings?.secondLayerStyleSettings,
            initialState = initialState,
            customLogo = bannerSettings?.generalStyleSettings?.logo,
            labels = uiHolder.data.settings.internationalizationLabels,
            theme = theme,
            coordinator = coordinator,
            toggleMediator = toggleMediator!!,
            landscapeMode = landscapeMode,
            linksSettings = linksSettings,
            statusBarColor = bannerSettings?.generalStyleSettings?.statusBarColor
        )

        bannerContainerView?.showSecondLayer(viewModel)
    }

    fun dismiss() {
        bannerContainerView?.cancelLogoDownload()
        bannerTransition?.exit(::dismissDialogEffectively) ?: dismissDialogEffectively()
        resetStatusBarColor()
    }

    private fun dismissDialogEffectively() {
        if (getContext()?.isActivityFinishingOrDestroyed() == true) {
            cleanUpResources()
            return
        }

        if (alertDialog?.isShowing == true) {
            alertDialog?.dismiss()
        }

        cleanUpResources()
    }

    private fun cleanUpResources() {
        toggleMediator?.tearDown()
        toggleMediator = null
        alertDialog = null
        bannerContainerView = null
        bannerTransition = null
    }

    private fun resetStatusBarColor() {
        val hasCustomizedStatusBar = bannerSettings?.generalStyleSettings?.statusBarColor != null
        if (!hasCustomizedStatusBar) {
            return
        }

        val window = (getContext() as? Activity)?.window ?: return
        if (activityStatusBarColor != null) {
            window.statusBarColor = activityStatusBarColor
        }
    }
}
