package com.usercentrics.sdk.ui.banner

import android.animation.LayoutTransition
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.graphics.drawable.GradientDrawable
import android.view.Gravity
import android.widget.FrameLayout
import androidx.annotation.ColorInt
import androidx.core.view.children
import com.usercentrics.sdk.PopupPosition
import com.usercentrics.sdk.UsercentricsLayout
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerView
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerViewModel
import com.usercentrics.sdk.ui.layerView.cancelLogoDownload
import com.usercentrics.sdk.ui.secondLayer.UCSecondLayerView
import com.usercentrics.sdk.ui.secondLayer.UCSecondLayerViewModel
import com.usercentrics.sdk.ui.theme.UCThemeData

@SuppressLint("ViewConstructor")
internal class UCBannerContainerView(
    private val context: Context,
    private val theme: UCThemeData,
    private val themedContext: Context,
    private val isFullScreen: Boolean,
) : FrameLayout(themedContext), UCBannerContainer {

    private val contentView = FrameLayout(themedContext).also {
        it.layoutTransition = LayoutTransition().apply {
            enableTransitionType(
                LayoutTransition.APPEARING or LayoutTransition.CHANGE_APPEARING
                        or LayoutTransition.CHANGE_DISAPPEARING or LayoutTransition.DISAPPEARING
                        or LayoutTransition.CHANGING
            )
        }
        addView(it)
    }

    override fun showFirstLayer(viewModel: UCFirstLayerViewModel, layout: UsercentricsLayout, customBackgroundColor: Int?, customCornerRadiusInDp: Int?) {
        val cornerRadius = (customCornerRadiusInDp ?: theme.bannerCornerRadius).toFloat().dpToPx(themedContext)
        styleLayout(
            layout = layout,
            layerBackgroundColor = customBackgroundColor ?: theme.colorPalette.layerBackgroundColor,
            cornerRadius = cornerRadius,
            statusBarColor = viewModel.statusBarColor,
        )

        contentView.removeAllViews()

        val firstLayerView = UCFirstLayerView(themedContext, theme, cornerRadius, viewModel)
        contentView.addView(firstLayerView)
    }

    override fun showSecondLayer(viewModel: UCSecondLayerViewModel) {
        styleFullScreen(theme.colorPalette.layerBackgroundColor)
        applyStatusBarColor(viewModel.statusBarColor)

        contentView.removeAllViews()
        val secondLayerView = UCSecondLayerView(themedContext, theme).also {
            it.bindViewModel(viewModel)
        }
        contentView.addView(secondLayerView)
    }

    override fun cancelLogoDownload() {
        when (val view = contentView.children.first()) {
            is UCFirstLayerView -> view.cancelLogoDownload(R.id.ucFirstLayerHeaderLogo)
            is UCSecondLayerView -> view.cancelLogoDownload(R.id.ucHeaderLogo)
            else -> assert(false) { "Unknown view type ${view.javaClass} types possible: UCFirstLayerView, UCSecondLayerView " }
        }
    }

    private fun styleLayout(layout: UsercentricsLayout, @ColorInt layerBackgroundColor: Int?, cornerRadius: Float, statusBarColor: Int?) {
        when (layout) {
            is UsercentricsLayout.Sheet -> styleBottomSheet(layerBackgroundColor, cornerRadius)
            is UsercentricsLayout.Popup -> stylePopup(layout, layerBackgroundColor, cornerRadius)
            is UsercentricsLayout.Full -> styleFullScreen(layerBackgroundColor)
        }
        applyStatusBarColor(statusBarColor)
    }

    private fun styleBottomSheet(@ColorInt layerBackgroundColor: Int?, cornerRadius: Float) {
        this.tag = Gravity.BOTTOM

        contentView.apply {
            background = GradientDrawable().apply {
                cornerRadii = listOf(cornerRadius, cornerRadius, cornerRadius, cornerRadius, 0f, 0f, 0f, 0f).toFloatArray()
                layerBackgroundColor?.let { setColor(it) }
            }

            layoutParams = LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT, Gravity.BOTTOM).apply {
                marginStart = 0
                marginEnd = 0
                topMargin = 24.dpToPx(themedContext)
            }
        }
    }

    private fun styleFullScreen(@ColorInt layerBackgroundColor: Int?) {
        layerBackgroundColor?.let { contentView.setBackgroundColor(it) }

        if (isFullScreen) {
            this.tag = Gravity.NO_GRAVITY
            contentView.layoutParams = LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT, Gravity.NO_GRAVITY)
        } else {
            this.tag = -1
            contentView.layoutParams = LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)
        }
    }

    private fun applyStatusBarColor(color: Int?) {
        val statusBarColor = color ?: return
        val window = (context as? Activity)?.window ?: return

        window.statusBarColor = statusBarColor
    }

    private fun stylePopup(layout: UsercentricsLayout.Popup, @ColorInt layerBackgroundColor: Int?, cornerRadius: Float) {
        val gradientDrawable = GradientDrawable().apply {
            this.cornerRadius = cornerRadius
            layerBackgroundColor?.let { setColor(it) }
        }
        contentView.background = gradientDrawable

        val gravity = when (layout.position) {
            PopupPosition.BOTTOM -> Gravity.BOTTOM
            PopupPosition.CENTER -> Gravity.CENTER
        }

        this.tag = gravity

        contentView.layoutParams = LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT, gravity).apply {
            val defaultMargin by lazy { resources.getDimensionPixelOffset(R.dimen.ucFirstLayerPopupMargin) }
            val horizontalMargin = layout.horizontalMarginInDp?.dpToPx(themedContext)?.toInt() ?: defaultMargin
            val verticalMargin = layout.verticalMarginInDp?.dpToPx(themedContext)?.toInt() ?: defaultMargin

            marginStart = horizontalMargin
            marginEnd = horizontalMargin
            topMargin = verticalMargin
            bottomMargin = verticalMargin
        }
    }
}

