package com.usercentrics.sdk.ui.components

import android.content.Context
import android.graphics.Color
import android.graphics.Typeface
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.GradientDrawable
import android.graphics.drawable.StateListDrawable
import android.util.AttributeSet
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.annotation.AttrRes
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.graphics.ColorUtils
import com.usercentrics.sdk.ButtonLayout
import com.usercentrics.sdk.ButtonSettings
import com.usercentrics.sdk.ButtonType
import com.usercentrics.sdk.models.settings.FirstLayerButtonLabels
import com.usercentrics.sdk.models.settings.PredefinedUIButtonType
import com.usercentrics.sdk.models.settings.PredefinedUIFooterButton
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.extensions.emptyToNull
import com.usercentrics.sdk.ui.theme.UCButtonCustomization
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class UCButton : ConstraintLayout {

    private val ucButtonBackground by lazy { findViewById<UCImageView>(R.id.ucButtonBackground) }
    private val ucButtonText: UCTextView by lazy { findViewById(R.id.ucButtonText) }

    var text: CharSequence
        get() = ucButtonText.text
        set(value) {
            ucButtonText.text = value
        }

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initLayout()
    }

    private fun initLayout() {
        LayoutInflater.from(context).inflate(R.layout.uc_button, this)
    }

    private fun setCustomBackground(color: Int, cornerRadius: Int) {
        val radiusPx = cornerRadius.dpToPx(context)

        // Tries to detect parent background
        val parentBg = ((parent as? ViewGroup)?.background as? ColorDrawable)?.color ?: Color.WHITE

        // If the button color has alpha, composite it over the parent bg first
        val effectiveFill = if (Color.alpha(color) < 255) {
            ColorUtils.compositeColors(color, parentBg)
        } else color

        val ringColor = pickFocusRingColor(effectiveFill, parentBg)

        fun shape(fill: Int, strokeColor: Int? = null, strokeWidthDp: Int = 4) =
            GradientDrawable().apply {
                setCornerRadius(radiusPx.toFloat())
                setColor(fill)
                strokeColor?.let { setStroke(strokeWidthDp.dpToPx(context), it) }
            }

        val normal = shape(effectiveFill)
        val focused = shape(effectiveFill, ringColor)

        background = StateListDrawable().apply {
            addState(intArrayOf(android.R.attr.state_focused), focused)
            addState(intArrayOf(), normal)
        }
    }

    private fun pickFocusRingColor(fill: Int, bg: Int): Int {
        val candidates = intArrayOf(Color.WHITE, Color.BLACK, Color.RED, Color.GREEN, Color.YELLOW, Color.BLUE)

        var bestCandidate = candidates[0]
        var bestMinContrast = -1.0
        for (candidate in candidates) {
            val contrastOnFill = ColorUtils.calculateContrast(candidate, fill)
            val contrastOnBackground = ColorUtils.calculateContrast(candidate, bg)
            val minContrast = minOf(contrastOnFill, contrastOnBackground)

            if (minContrast > bestMinContrast) {
                bestMinContrast = minContrast
                bestCandidate = candidate
            }
        }
        return bestCandidate
    }

    fun setup(settings: UCButtonSettings, onClick: () -> Unit) {
        text = settings.label
        setOnClickListener { onClick() }

        minimumHeight = 40.dpToPx(context)
        ucButtonText.letterSpacing = 0f

        if (settings.backgroundColor != null) {
            setCustomBackground(settings.backgroundColor, settings.cornerRadius)
        }

        ucButtonText.apply {
            typeface = settings.font
            setTextSize(TypedValue.COMPLEX_UNIT_SP, settings.textSizeInSp)
            isAllCaps = settings.isAllCaps
            if (settings.textColor != null) {
                setTextColor(settings.textColor)
            }
        }
    }
}


internal data class UCButtonSettings(
    val label: String,
    val backgroundColor: Int?,
    val cornerRadius: Int,
    val textColor: Int?,
    val textSizeInSp: Float,
    val isAllCaps: Boolean,
    val type: UCButtonType,
    val font: Typeface,
) {
    companion object {
        fun map(
            isCCPA: Boolean,
            buttonLayout: ButtonLayout?,
            defaultButtons: List<List<PredefinedUIFooterButton>>,
            theme: UCThemeData,
            buttonLabels: FirstLayerButtonLabels
        ): List<List<UCButtonSettings>> {
            val defaultButtonsProcessed by lazy { defaultButtons.map { row -> row.map { map(it, theme) } } }

            // CCPA Safeguard
            if (isCCPA) {
                return defaultButtonsProcessed
            }

            return when (buttonLayout) {
                is ButtonLayout.Column -> {
                    val buttons = buttonLayout.buttons.emptyToNull()?.map { map(it, theme, buttonLabels) }
                        ?: defaultButtons.flatten().map { map(it, theme) }
                    buttons.map { listOf(it) }
                }
                is ButtonLayout.Row -> {
                    val buttons = buttonLayout.buttons.emptyToNull()?.map { map(it, theme, buttonLabels) }
                        ?: defaultButtons.flatten().map { map(it, theme) }
                    listOf(buttons)
                }
                is ButtonLayout.Grid -> {
                    buttonLayout.buttons.emptyToNull()?.map { row -> row.map { map(it, theme, buttonLabels) } }
                        ?: defaultButtons.map { row -> row.map { map(it, theme) } }
                }
                null -> defaultButtonsProcessed
            }
        }

        fun map(predefinedUIButton: PredefinedUIFooterButton, theme: UCThemeData): UCButtonSettings {
            val buttonType = UCButtonType.from(predefinedUIButton.type)
            val customization = buttonType.getCustomization(theme)
            return UCButtonSettings(
                label = predefinedUIButton.label,
                backgroundColor = customization.background,
                cornerRadius = customization.cornerRadius,
                isAllCaps = false,
                textColor = customization.text,
                type = buttonType,
                font = theme.fonts.fontBold,
                textSizeInSp = theme.fonts.sizes.body
            )
        }

        fun map(button: ButtonSettings, theme: UCThemeData, buttonLabels: FirstLayerButtonLabels): UCButtonSettings {
            val buttonType = UCButtonType.from(button.type)
            val fallbackCustomization = buttonType.getCustomization(theme)
            return UCButtonSettings(
                label = button.type.getLabel(buttonLabels),
                backgroundColor = button.backgroundColor ?: fallbackCustomization.background,
                cornerRadius = button.cornerRadius ?: fallbackCustomization.cornerRadius,
                isAllCaps = button.isAllCaps ?: false,
                textColor = button.textColor ?: fallbackCustomization.text,
                type = buttonType,
                font = button.font ?: theme.fonts.fontBold,
                textSizeInSp = button.textSizeInSp ?: theme.fonts.sizes.body
            )
        }
    }
}

internal enum class UCButtonType {
    ACCEPT_ALL,
    DENY_ALL,
    SAVE,
    MORE,
    OK;

    companion object {
        fun from(type: ButtonType) = when (type) {
            ButtonType.ACCEPT_ALL -> ACCEPT_ALL
            ButtonType.DENY_ALL -> DENY_ALL
            ButtonType.MORE -> MORE
            ButtonType.SAVE -> SAVE
        }

        fun from(type: PredefinedUIButtonType) = when (type) {
            PredefinedUIButtonType.ACCEPT_ALL -> ACCEPT_ALL
            PredefinedUIButtonType.DENY_ALL -> DENY_ALL
            PredefinedUIButtonType.SAVE_SETTINGS -> SAVE
            PredefinedUIButtonType.MANAGE_SETTINGS -> MORE
            PredefinedUIButtonType.OK -> OK
        }
    }
}

private fun ButtonType.getLabel(buttonLabels: FirstLayerButtonLabels): String {
    return when (this) {
        ButtonType.ACCEPT_ALL -> buttonLabels.acceptAll
        ButtonType.DENY_ALL -> buttonLabels.denyAll
        ButtonType.MORE -> buttonLabels.more
        ButtonType.SAVE -> buttonLabels.save
    }
}

private fun UCButtonType.getCustomization(theme: UCThemeData): UCButtonCustomization {
    return when (this) {
        UCButtonType.ACCEPT_ALL -> theme.buttonTheme.acceptAll
        UCButtonType.DENY_ALL -> theme.buttonTheme.denyAll
        UCButtonType.MORE -> theme.buttonTheme.manage
        UCButtonType.SAVE -> theme.buttonTheme.save
        UCButtonType.OK -> theme.buttonTheme.ok
    }
}
