package com.usercentrics.sdk.ui.components

import android.content.Context
import android.graphics.*
import android.util.AttributeSet
import androidx.annotation.AttrRes
import androidx.appcompat.widget.AppCompatImageView
import com.usercentrics.sdk.UsercentricsImage
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.image.UCRemoteImage
import com.usercentrics.sdk.ui.image.UCRemoteImageService
import com.usercentrics.sdk.ui.theme.UCThemeData
import kotlinx.coroutines.*

class UCImageView : AppCompatImageView {

    private val remoteImageService: UCRemoteImageService by lazy { PredefinedUIDependencyManager.remoteImageService.value }
    private val logger: UsercentricsLogger? by lazy { PredefinedUIDependencyManager.logger }
    var cornerSettings: CornerRadiusSettings? = null
    private var job: Job? = null

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr)

    fun setImage(image: UsercentricsImage) {
        when (image) {
            is UsercentricsImage.ImageDrawableId -> setImageResource(image.drawableResId)
            is UsercentricsImage.ImageBitmap -> setImageBitmap(image.bitmap)
            is UsercentricsImage.ImageUrl -> setImageUrl(image.imageUrl)
            is UsercentricsImage.ImageDrawable -> setImageDrawable(image.drawable)
        }
    }

    fun cancelJob() {
        job?.cancel()
    }

    private fun setImageUrl(imageUrl: String) {
        job = MainScope().launch {
            val remoteImage = tryToDownloadImage(imageUrl) ?: return@launch

            yield()

            if (remoteImage.isSVG()) {
                tryToSetImageSVG(imageUrl, remoteImage.payload)
            } else {
                tryToSetImageBitmap(imageUrl, remoteImage.payload)
            }
        }
    }

    private suspend fun tryToDownloadImage(imageUrl: String): UCRemoteImage? {
        return runCatching {

            yield()

            downloadImage(imageUrl)
        }.onFailure {

            yield()

            logger?.error("Error when loading image with URL<$imageUrl>, please make sure that you are proving a https:// URL in the Admin Interface", it)
        }.getOrNull()
    }

    private suspend fun downloadImage(imageUrl: String): UCRemoteImage = withContext(Dispatchers.IO) {
        remoteImageService.getImage(imageUrl)
    }

    private suspend fun tryToSetImageSVG(imageUrl: String, payload: ByteArray) {
        runCatching {
            setImageSVG(String(payload))
        }.onFailure {
            logger?.svgError(imageUrl, it)
        }
    }

    private suspend fun setImageSVG(svg: String) = withContext(Dispatchers.IO) {
        com.pixplicity.sharp.Sharp.loadString(svg).into(this@UCImageView)
    }

    private fun UsercentricsLogger.svgError(imageUrl: String, cause: Throwable) {
        val isSvgModuleNotPresent = cause is NoClassDefFoundError
        if (isSvgModuleNotPresent) {
            error("Error when trying to use image with URL<$imageUrl> as a SVG because the optional SVG module is not present. Please add this module to your application: 'com.pixplicity.sharp'")
        } else {
            error("Error when trying to use image with URL<$imageUrl> as a SVG", cause)
        }
    }

    private suspend fun tryToSetImageBitmap(imageUrl: String, payload: ByteArray) {
        runCatching {
            setImageBitmap(decodeBitmap(payload))
        }.onFailure {
            logger?.error("Error when trying to use image with URL<$imageUrl> as a Bitmap", it)
        }
    }

    private suspend fun decodeBitmap(bytes: ByteArray): Bitmap = withContext(Dispatchers.IO) {
        BitmapFactory.decodeByteArray(bytes, 0, bytes.size) ?: throw IllegalStateException("Cannot decode the image byte array as a Bitmap")
    }

    fun styleIcon(theme: UCThemeData) {
        theme.colorPalette.text80?.let {
            setColorFilter(it, PorterDuff.Mode.SRC_IN)
        }
    }

    override fun onDraw(canvas: Canvas) {
        cornerSettings?.getPath(width.toFloat(), height.toFloat())?.let {
            canvas.clipPath(it)
        }
        super.onDraw(canvas)
    }

    class CornerRadiusSettings(
        private val topLeft: Float? = null,
        private val topRight: Float? = null,
        private val bottomRight: Float? = null,
        private val bottomLeft: Float? = null
    ) {
        fun getPath(width: Float, height: Float): Path {
            val radii = FloatArray(8)
            if (topLeft != null) {
                radii[0] = topLeft
                radii[1] = topLeft
            }
            if (topRight != null) {
                radii[2] = topRight
                radii[3] = topRight
            }
            if (bottomRight != null) {
                radii[4] = bottomRight
                radii[5] = bottomRight
            }
            if (bottomLeft != null) {
                radii[6] = bottomLeft
                radii[7] = bottomLeft
            }
            return Path().apply {
                addRoundRect(
                    RectF(0f, 0f, width, height),
                    radii, Path.Direction.CW
                )
            }
        }
    }
}
