package com.usercentrics.sdk.ui.components.cards

import android.content.Context
import android.graphics.Color
import android.graphics.drawable.Drawable
import android.graphics.drawable.GradientDrawable
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import androidx.annotation.AttrRes
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.constraintlayout.widget.ConstraintSet
import com.usercentrics.sdk.models.settings.PredefinedUIAriaLabels
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.accessibility.accessibleTouchTarget
import com.usercentrics.sdk.ui.components.UCButton
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.components.UCToggle
import com.usercentrics.sdk.ui.components.UCTogglePM
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getExpandIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.styleIcon
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.extensions.setMarginBottom
import com.usercentrics.sdk.ui.extensions.setMarginTop
import com.usercentrics.sdk.ui.extensions.setPaddingBottom
import com.usercentrics.sdk.ui.secondLayer.UCCardsContentPM
import com.usercentrics.sdk.ui.theme.UCColorPalette
import com.usercentrics.sdk.ui.theme.UCThemeData

internal class UCCard : LinearLayout {

    private val cardDefaultMargin by lazy {
        context.resources.getDimension(R.dimen.ucCardVerticalMargin).toInt()
    }

    private val ucCardTitle by lazy { findViewById<UCTextView>(R.id.ucCardTitle) }
    private val ucCardSwitch by lazy { findViewById<UCToggle>(R.id.ucCardSwitch) }
    private val ucCardIcon by lazy { findViewById<UCButton>(R.id.ucCardIcon) }
    private val ucCardDescription by lazy { findViewById<UCTextView>(R.id.ucCardDescription) }
    private val ucCardExpandableContent by lazy { findViewById<ViewGroup>(R.id.ucCardExpandableContent) }
    private val ucCardHeader by lazy { findViewById<View>(R.id.ucCardHeader) }
    private val ucCardSwitchListDivider by lazy { findViewById<View>(R.id.ucCardSwitchListDivider) }
    private val ucCardSwitchList by lazy { findViewById<ViewGroup>(R.id.ucCardSwitchList) }
    private val ucCardBottomSpacing by lazy { findViewById<View>(R.id.ucCardBottomSpacing) }
    private val ucCardDividerExpandedContent by lazy { findViewById<View>(R.id.ucCardDividerExpandedContent) }

    private val expandIconDrawable: Drawable? by lazy { context.getExpandIcon() }

    private val ariaLabels: PredefinedUIAriaLabels by lazy { PredefinedUIDependencyManager.ariaLabels }

    private var isExpanded = false
    private var onExpandHandler: (Boolean) -> Unit = {}
    var onExpandedListener: UCCardExpandedListener = { _, _ -> }

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initLayout(context)
    }

    private fun initLayout(context: Context) {
        inflate(context, R.layout.uc_card, this)
        ucCardIcon.background = expandIconDrawable
        orientation = VERTICAL
    }

    fun style(theme: UCThemeData) {
        background = theme.colorPalette.getCardBackground(context)

        ucCardTitle.styleBody(theme, isBold = true)
        ucCardDescription.styleBody(theme)
        ucCardSwitch.styleToggle(theme)

        ucCardSwitchListDivider.setBackgroundColor(theme.colorPalette.tabsBorderColor)
        ucCardDividerExpandedContent.setBackgroundColor(theme.colorPalette.tabsBorderColor)

        expandIconDrawable?.styleIcon(theme)
    }

    fun bindCard(
        theme: UCThemeData,
        model: UCCardPM,
        isExpanded: Boolean,
        onExpandHandler: ((Boolean) -> Unit)?,
        onMoreInfo: ((String) -> Unit)?
    ) {
        ucCardTitle.text = model.title.trim()
        ucCardTitle.labelFor = ucCardSwitch.id

        val description = model.description?.trim() ?: ""
        ucCardDescription.text = description

        val descriptionIsVisible = description.isNotBlank()
        toggleCardDescriptionVisibility(isVisible = descriptionIsVisible)

        toggleBottomSpacingVisibility(isVisible = !descriptionIsVisible)

        bindMainToggle(model)

        val toggleList = model.toggleList
        if (toggleList.isNullOrEmpty()) {
            removeTogglesList()
        } else {
            bindTogglesList(theme, toggleList)
        }

        this.onExpandHandler = onExpandHandler ?: { expanded ->
            updateExpandableContent(theme, model, onMoreInfo)
            if (expanded) {
                post {
                    val cardLocation = intArrayOf(0, 0)
                    getLocationOnScreen(cardLocation)
                    onExpandedListener(cardLocation[1], height)
                }
            }
        }
        this.isExpanded = isExpanded
        ucCardExpandableContent.removeAllViews()
        updateExpandableContent(theme, model, onMoreInfo)
        setExpandableInteraction(model)
    }

    private fun setExpandableInteraction(model: UCCardPM) {
        val hasExpandableContent = hasExpandableContent(model)
        setCardClickable(hasExpandableContent)

        if (!hasExpandableContent) {
            ucCardIcon.visibility = View.GONE
            return
        }

        ucCardIcon.visibility = View.VISIBLE
        ucCardHeader.setOnClickListener {
            expandCategoryServices()
        }
        ucCardIcon.setOnClickListener {
            expandCategoryServices()
        }
    }

    private fun expandCategoryServices() {
        isExpanded = !isExpanded
        onExpandHandler(isExpanded)
    }

    private fun updateExpandableContent(theme: UCThemeData, model: UCCardPM, onMoreInfo: ((String) -> Unit)?) {
        val expandButtonAriaLabel: String?

        if (isExpanded) {
            ucCardIcon.rotation = 180f
            ucCardHeader.setMarginBottom(cardDefaultMargin)

            addExpandableContentView(theme, model, onMoreInfo)

            toggleCardDescriptionPadding(hasPadding = false)
            toggleDividerForExpandedContent(isVisible = true)

            expandButtonAriaLabel = ariaLabels.collapse
        } else {
            ucCardExpandableContent.removeAllViews()
            ucCardIcon.rotation = 0f
            ucCardHeader.setMarginBottom(0)

            toggleCardDescriptionPadding(hasPadding = true)
            toggleDividerForExpandedContent(isVisible = false)

            expandButtonAriaLabel = ariaLabels.expand
        }

        val cardDescriptionLabel = "$expandButtonAriaLabel ${model.title} ${ariaLabels.usercentricsCard}"
        val iconDescriptionLabel = "$expandButtonAriaLabel ${model.title.trim()}"

        ucCardHeader.contentDescription = cardDescriptionLabel
        ucCardIcon.contentDescription = iconDescriptionLabel

        post {
            ucCardHeader.accessibleTouchTarget()
            ucCardIcon.accessibleTouchTarget()
        }
    }

    private fun toggleDividerForExpandedContent(isVisible: Boolean) {
        ucCardDividerExpandedContent.visibility = if (isVisible) View.VISIBLE else View.GONE
    }

    private fun setCardClickable(isClickable: Boolean) {
        ucCardHeader.apply {
            this.isClickable = isClickable
            this.isFocusable = isClickable
        }

        ucCardIcon.apply {
            this.isClickable = isClickable
            this.isFocusable = isClickable
        }
    }

    private fun toggleCardDescriptionPadding(hasPadding: Boolean) {
        val padding = if (hasPadding) cardDefaultMargin else 0
        ucCardDescription.setPaddingBottom(padding)
    }

    private fun toggleCardDescriptionVisibility(isVisible: Boolean) {
        ucCardDescription.visibility = if (isVisible) VISIBLE else GONE
    }

    private fun toggleBottomSpacingVisibility(isVisible: Boolean) {
        ucCardBottomSpacing.visibility = if (isVisible) View.VISIBLE else GONE
    }

    private fun addExpandableContentView(theme: UCThemeData, model: UCCardPM, onMoreInfo: ((String) -> Unit)?) {
        if (!hasExpandableContent(model)) {
            return
        }
        val smallView = UCCardSections(context).apply { bind(theme, model.contentSections, onMoreInfo) }
        ucCardExpandableContent.addView(smallView)
    }

    private fun hasExpandableContent(model: UCCardPM): Boolean {
        return model.contentSections.isNotEmpty()
    }

    private fun bindMainToggle(model: UCCardPM) {
        val mainToggle = model.mainToggle
        if (mainToggle != null) {
            ucCardSwitch.bindLegacy(mainToggle)
            ucCardSwitch.visibility = View.VISIBLE
        } else {
            ucCardSwitch.visibility = View.GONE
        }
    }

    private fun bindTogglesList(theme: UCThemeData, togglesList: List<UCTogglePM>) {
        ucCardSwitchList.removeAllViews()
        toggleCardSwitchListVisibility(isVisible = true)

        for (toggleModel in togglesList) {
            val cardExtraSwitch = LayoutInflater.from(context).inflate(R.layout.uc_card_extra_switch, null)
            cardExtraSwitch.findViewById<UCTextView>(R.id.ucCardSwitchText).apply {
                text = toggleModel.label
                theme.colorPalette.text100?.let { setTextColor(it) }
            }
            cardExtraSwitch.findViewById<UCToggle>(R.id.ucCardSwitchInside).apply {
                contentDescription = toggleModel.label
                styleToggle(theme)
                bindLegacy(toggleModel)
            }

            ucCardSwitchList.addView(cardExtraSwitch)
            post {
                cardExtraSwitch.accessibleTouchTarget()
            }
        }
    }

    private fun removeTogglesList() {
        ucCardSwitchList.removeAllViews()
        toggleCardSwitchListVisibility(isVisible = false)
    }

    private fun toggleCardSwitchListVisibility(isVisible: Boolean) {
        val visibility = if (isVisible) View.VISIBLE else GONE
        ucCardSwitchList.visibility = visibility
        ucCardSwitchListDivider.visibility = visibility
        ucCardDividerExpandedContent.setMarginTop(if (isVisible) 0 else cardDefaultMargin)

        val constraintTargetView = if (isVisible) ucCardSwitchList else ucCardTitle
        (ucCardIcon.layoutParams as ConstraintLayout.LayoutParams).apply {
            bottomToBottom = constraintTargetView.id
            topToTop = constraintTargetView.id
            endToEnd = ConstraintSet.PARENT_ID
        }
    }
}

internal sealed class UCCardComponent {
    companion object {
        fun from(content: List<UCCardsContentPM>): List<UCCardComponent> {
            val result = mutableListOf<UCCardComponent>()
            for (cardsContent in content) {
                val title = cardsContent.title
                if (!title.isNullOrBlank()) {
                    result.add(UCSectionTitlePM(title))
                }

                result.addAll(cardsContent.cards)

                val controllerId = cardsContent.controllerId
                if (controllerId != null) {
                    result.add(controllerId)
                }
            }
            return result
        }
    }
}

internal class UCCardPM(
    val id: String,
    val title: String,
    val description: String?,
    val mainToggle: UCTogglePM?,
    val contentSections: List<UCContentSectionPM>,
    val toggleList: List<UCTogglePM>?
) : UCCardComponent()

internal data class UCSectionTitlePM(val title: String) : UCCardComponent()

internal class UCControllerIdPM(
    val label: String,
    val value: String,
    val ariaLabel: String,
    val onCopyControllerId: () -> Unit,
) : UCCardComponent()

internal typealias UCCardExpandedListener = (cardPositionY: Int, cardHeight: Int) -> Unit

internal fun UCColorPalette.getCardBackground(context: Context): Drawable {
    return GradientDrawable().apply {
        shape = GradientDrawable.RECTANGLE
        cornerRadius = 4.dpToPx(context).toFloat()
        setStroke(1.dpToPx(context), tabsBorderColor)
        setColor(layerBackgroundColor ?: Color.WHITE)
    }
}
