package com.usercentrics.sdk.ui.components.cookie

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.graphics.drawable.GradientDrawable
import android.view.View
import android.widget.LinearLayout
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.usercentrics.sdk.models.settings.PredefinedUIDeviceStorageContent
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCImageView
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getCloseIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.styleIcon
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.extensions.inflater
import com.usercentrics.sdk.ui.theme.UCThemeData

@SuppressLint("ViewConstructor")
internal class UCCookiesView(
    context: Context,
    private val theme: UCThemeData,
    private val viewModel: UCCookiesViewModel
) : ConstraintLayout(context) {

    private val ucCookieDialogTitle: UCTextView by lazy { findViewById(R.id.ucCookieDialogTitle) }
    private val ucCookieLoadingText: UCTextView by lazy { findViewById(R.id.ucCookieLoadingText) }
    private val ucCookieTryAgainBtn: UCTextView by lazy { findViewById(R.id.ucCookieTryAgainBtn) }
    private val ucCookieRetryMessage: UCTextView by lazy { findViewById(R.id.ucCookieRetryMessage) }
    private val ucCookieLoadingBox: LinearLayout by lazy { findViewById(R.id.ucCookieLoadingBox) }
    private val ucCookieRetryBox: LinearLayout by lazy { findViewById(R.id.ucCookieRetryBox) }
    private val ucCookieDialogList: RecyclerView by lazy { findViewById(R.id.ucCookieDialogList) }
    private val ucCookieDialogClose: UCImageView by lazy { findViewById(R.id.ucCookieDialogClose) }

    init {
        setPadding()
        inflateView()
        applyTheme()
        bindContent()
    }

    private fun setPadding() {
        val padding = 12.dpToPx(context)
        setPaddingRelative(padding, padding, padding, padding)
    }

    private fun inflateView() {
        context.inflater.inflate(R.layout.uc_cookie_dialog, this)
    }

    private fun applyTheme() {
        ucCookieDialogTitle.styleBody(theme, isBold = true)
        ucCookieLoadingText.styleBody(theme)
        ucCookieTryAgainBtn.styleBody(theme, isLink = true)
        ucCookieRetryMessage.styleBody(theme)

        val icon = context.getCloseIcon()?.also {
            it.styleIcon(theme)
        }
        ucCookieDialogClose.setImageDrawable(icon)

        theme.colorPalette.layerBackgroundSecondaryColor?.let { setBackgroundColor(it) }

        ucCookieRetryBox.background = cookieInformationBoxBackground()
        ucCookieLoadingBox.background = cookieInformationBoxBackground()
    }

    private fun cookieInformationBoxBackground(): GradientDrawable {
        return GradientDrawable().apply {
            shape = GradientDrawable.RECTANGLE
            setColor(theme.colorPalette.layerBackgroundSecondaryColor ?: Color.WHITE)
            setStroke(1.dpToPx(context), theme.colorPalette.tabsBorderColor)
        }
    }

    private fun bindContent() {
        ucCookieDialogTitle.text = viewModel.titleDetailed
        ucCookieLoadingText.text = viewModel.loading
        ucCookieRetryMessage.text = viewModel.error
        ucCookieTryAgainBtn.text = viewModel.tryAgain
        ucCookieDialogClose.setOnClickListener { viewModel.onDismiss() }

        loadInformation()
    }

    private fun loadInformation() {
        ucCookieLoadingBox.visibility = View.VISIBLE
        ucCookieRetryBox.visibility = View.GONE
        ucCookieDialogList.visibility = View.GONE

        viewModel.loadInformation(onSuccess = { disclosures ->
            showCookieInfo(disclosures)
        }, onError = {
            showRetry()
        })
    }

    private fun showRetry() {
        ucCookieLoadingBox.visibility = View.GONE
        ucCookieDialogList.visibility = View.GONE
        ucCookieRetryBox.visibility = View.VISIBLE

        ucCookieTryAgainBtn.setOnClickListener {
            loadInformation()
        }
    }

    private fun showCookieInfo(data: List<PredefinedUIDeviceStorageContent>) {
        ucCookieLoadingBox.visibility = View.GONE
        ucCookieRetryBox.visibility = View.GONE
        ucCookieDialogList.visibility = View.VISIBLE

        val listAdapter = CookieInformationAdapter(theme, data)
        ucCookieDialogList.adapter = listAdapter
        ucCookieDialogList.layoutManager = LinearLayoutManager(context)
    }
}
