package com.usercentrics.sdk.ui.firstLayer

import android.annotation.SuppressLint
import android.content.Context
import android.view.View
import android.widget.FrameLayout
import androidx.appcompat.widget.LinearLayoutCompat
import androidx.core.widget.NestedScrollView
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.firstLayer.component.*
import com.usercentrics.sdk.ui.theme.UCThemeData

@SuppressLint("ViewConstructor")
internal class UCFirstLayerView(
    context: Context,
    theme: UCThemeData,
    cornerRadius: Float,
    viewModel: UCFirstLayerViewModel
) : LinearLayoutCompat(context) {

    private val cardsVerticalMargin by lazy { resources.getDimensionPixelOffset(R.dimen.ucFirstLayerCardsVerticalMargin) }
    private val scrollView by lazy {
        NestedScrollView(context).apply {
            id = R.id.ucBannerFirstLayerScrollContainer
            orientation = VERTICAL
            isFillViewport = true
        }
    }
    private val scrollableContainer by lazy {
        LinearLayoutCompat(context).apply {
            orientation = VERTICAL
        }
    }
    private val footerSeparator: View

    init {
        setupView()

        if (viewModel.isCloseAfterHeaderImage()) {
            scrollableContainer.addHeaderImage(cornerRadius, viewModel)
            addCloseViewIfNeeded(theme, viewModel)
        } else {
            addCloseViewIfNeeded(theme, viewModel)
            scrollableContainer.addHeaderImage(cornerRadius, viewModel)
        }

        scrollableContainer.addTitle(theme, viewModel.title)
        scrollableContainer.addMessageAndReadMoreButton(theme, viewModel)
        scrollableContainer.addLinks(theme, viewModel)
        scrollableContainer.addContent(cardsVerticalMargin, theme, viewModel, ::onExpandedCardListener)
        scrollableContainer.addSpacing(viewModel)

        footerSeparator = this.addSeparator(theme)

        this.addCCPAToggle(theme, viewModel)
        this.addButtons(viewModel)
        this.addPoweredBy(theme, viewModel)
    }

    private fun addCloseViewIfNeeded(theme: UCThemeData, viewModel: UCFirstLayerViewModel) {
        when {
            viewModel.closeLink != null -> scrollableContainer.addContinueWithoutAccepting(theme, viewModel)
            viewModel.closeIcon -> scrollableContainer.addCloseButton(theme, viewModel)
        }
    }

    private fun setupView() {
        orientation = VERTICAL

        this.addView(scrollView, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT, 1f))

        scrollView.addView(scrollableContainer, FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT))
    }

    override fun onLayout(changed: Boolean, l: Int, t: Int, r: Int, b: Int) {
        super.onLayout(changed, l, t, r, b)
        val isScrollable = scrollView.height < scrollableContainer.height
        footerSeparator.visibility = if (isScrollable) VISIBLE else GONE
    }

    private fun onExpandedCardListener(cardPositionY: Int, cardHeight: Int) {
        val scrollViewLocation = intArrayOf(0, 0)
        scrollView.getLocationOnScreen(scrollViewLocation)
        val scrollViewPositionY = scrollViewLocation[1]

        val cardBottom = cardPositionY + cardHeight
        val scrollViewBottom = scrollViewPositionY + scrollView.height
        if (cardBottom > scrollViewBottom) {
            val dy = cardBottom - scrollViewBottom + cardsVerticalMargin
            scrollView.smoothScrollBy(0, dy)
        }
    }

}

private fun LinearLayoutCompat.addSpacing(viewModel: UCFirstLayerViewModel) {
    if (viewModel.useAllAvailableVerticalSpace) {
        this.addView(
            View(context).apply { visibility = View.INVISIBLE },
            LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, LinearLayoutCompat.LayoutParams.MATCH_PARENT, 100f)
        )
    }
}

private fun LinearLayoutCompat.addSeparator(theme: UCThemeData): View {
    val separatorView = View(context).apply {
        visibility = View.GONE
    }
    separatorView.setBackgroundColor(theme.colorPalette.tabsBorderColor)

    this.addView(
        separatorView,
        LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, 1.dpToPx(context))
    )
    return separatorView
}

