package com.usercentrics.sdk.ui.firstLayer.component

import android.content.res.Resources
import android.view.View
import android.widget.ImageView
import androidx.appcompat.widget.LinearLayoutCompat
import com.usercentrics.sdk.HeaderImageSettings
import com.usercentrics.sdk.SectionAlignment
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.UCImageView
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.firstLayer.UCFirstLayerViewModel

private val defaultLogoAlignment = SectionAlignment.START

internal fun LinearLayoutCompat.addHeaderImage(cornerRadius: Float, viewModel: UCFirstLayerViewModel?) {
    val hasClose = viewModel?.closeLink != null || viewModel?.closeIcon == true
    val imageAltTag = viewModel?.ariaLabels?.logoAltTag

    when (val headerImage = viewModel?.headerImage) {
        is HeaderImageSettings.ExtendedLogoSettings -> addExtendedLogo(cornerRadius, headerImage, hasClose, imageAltTag)
        is HeaderImageSettings.LogoSettings -> addLogo(headerImage, hasClose, imageAltTag)
        HeaderImageSettings.Hidden, null -> addSpacing(hasClose)
    }
}

private fun LinearLayoutCompat.addExtendedLogo(cornerRadius: Float, settings: HeaderImageSettings.ExtendedLogoSettings, hasClose: Boolean, imageAltTag: String?) {
    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, LinearLayoutCompat.LayoutParams.WRAP_CONTENT).apply {
        val margins = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)
        marginStart = 0
        marginEnd = 0
        topMargin = 0
        bottomMargin = if (hasClose) 0 else margins
    }
    val screenHeight: Int = Resources.getSystem().displayMetrics.heightPixels

    val view = UCImageView(context).apply {
        id = R.id.ucFirstLayerHeaderLogo
        setImage(settings.image)
        adjustViewBounds = true
        scaleType = ImageView.ScaleType.CENTER_CROP
        contentDescription = imageAltTag
        cornerSettings = UCImageView.CornerRadiusSettings(
            topLeft = cornerRadius,
            topRight = cornerRadius,
        )
        if (screenHeight > 0) {
            maxHeight = screenHeight / 3
        }
    }
    addView(view, layoutParams)
}

private fun LinearLayoutCompat.addLogo(settings: HeaderImageSettings.LogoSettings, hasClose: Boolean, imageAltTag: String?) {
    val alignment = settings.alignment ?: defaultLogoAlignment
    val height = settings.heightInDp?.dpToPx(context)?.toInt() ?: resources.getDimensionPixelOffset(R.dimen.ucFirstLayerDefaultLogoHeight)
    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, height).apply {
        val margins = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)

        marginStart = margins
        marginEnd = margins
        topMargin = if (hasClose) 0 else margins
        bottomMargin = margins
    }
    val view = UCImageView(context).apply {
        id = R.id.ucFirstLayerHeaderLogo
        setImage(settings.image)
        scaleType = alignment.toScaleType()
        contentDescription = imageAltTag
    }
    addView(view, layoutParams)
}

private fun LinearLayoutCompat.addSpacing(hasClose: Boolean) {
    val layoutParams = LinearLayoutCompat.LayoutParams(LinearLayoutCompat.LayoutParams.MATCH_PARENT, height).apply {
        val margins = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerInnerPadding)

        marginStart = 0
        topMargin = 0
        marginEnd = 0
        bottomMargin = if (hasClose) 0 else margins
    }
    val view = View(context).apply {
        visibility = View.INVISIBLE
    }
    addView(view, layoutParams)
}

private fun SectionAlignment.toScaleType(): ImageView.ScaleType {
    return when (this) {
        SectionAlignment.START -> ImageView.ScaleType.FIT_START
        SectionAlignment.CENTER -> ImageView.ScaleType.FIT_CENTER
        SectionAlignment.END -> ImageView.ScaleType.FIT_END
    }
}
