package com.usercentrics.sdk.ui.mappers

import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.ui.components.UCTogglePM
import com.usercentrics.sdk.ui.components.cards.UCCardPM
import com.usercentrics.sdk.ui.components.cards.UCContentSectionPM
import com.usercentrics.sdk.ui.components.cards.UCContentTextSectionPM
import com.usercentrics.sdk.ui.components.links.UCLinkPMLegacy
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleGroup
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediator
import com.usercentrics.sdk.v2.settings.data.DpsDisplayFormat

internal interface UCServiceMapper {
    fun map(
        service: PredefinedUICardUI,
        mainGroup: PredefinedUIToggleGroup?,
        toggleMediator: PredefinedUIToggleMediator,
        labels: PredefinedUILabels
    ): UCCardPM
}

internal class UCServiceMapperImpl(
    onOpenUrl: (String) -> Unit,
    onShowCookiesDialog: (PredefinedUIStorageInformationButtonInfo) -> Unit
) : UCServiceMapper {

    private val sectionMapper = UCServiceSectionMapper(onOpenUrl, onShowCookiesDialog)

    override fun map(
        service: PredefinedUICardUI,
        mainGroup: PredefinedUIToggleGroup?,
        toggleMediator: PredefinedUIToggleMediator,
        labels: PredefinedUILabels
    ): UCCardPM {
        return when (service.content) {
            is PredefinedUISingleServiceCardContent -> createServiceCard(service, labels, mainGroup, toggleMediator)
            is PredefinedUISimpleCardContent -> createSimpleCard(service)
            else -> throw NotImplementedError("Not supported card content")
        }
    }

    private fun createServiceCard(
        service: PredefinedUICardUI,
        labels: PredefinedUILabels,
        mainGroup: PredefinedUIToggleGroup?,
        toggleMediator: PredefinedUIToggleMediator
    ): UCCardPM {
        val content = service.content as PredefinedUISingleServiceCardContent
        val serviceDetails = content.service
        val serviceContentSection = serviceDetails.serviceContentSection

        val contentSectionsPM: List<UCContentSectionPM> =
            if (serviceContentSection != null) {
                contentSectionsFrom(serviceContentSection)
            } else {
                contentSectionsFrom(serviceDetails, labels)
            }

        val mainSwitchSettings = service.mainSwitchSettings
        val mainToggle = if (mainSwitchSettings != null) {
            UCTogglePM(mainSwitchSettings, group = mainGroup)
        } else {
            null
        }

        val toggleList = service.switchSettings?.map { itemSwitchSettings ->
            val group = toggleMediator.getServiceGroupLegacy(service.id, itemSwitchSettings)
            UCTogglePM(itemSwitchSettings, group)
        }

        return UCCardPM(
            id = service.id,
            title = service.title,
            description = service.shortDescription,
            mainToggle = mainToggle,
            contentSections = contentSectionsPM,
            toggleList = toggleList
        )
    }

    private fun createSimpleCard(predefinedCardUi: PredefinedUICardUI): UCCardPM {
        val description = (predefinedCardUi.content as PredefinedUISimpleCardContent).description

        return UCCardPM(
            id = predefinedCardUi.id,
            title = predefinedCardUi.title,
            description = predefinedCardUi.shortDescription,
            mainToggle = null,
            contentSections = listOf(UCContentTextSectionPM(title = null, description = description)),
            toggleList = null
        )
    }

    private fun contentSectionsFrom(
        serviceDetails: PredefinedUIServiceDetails,
        labels: PredefinedUILabels
    ): List<UCContentSectionPM> {
        val storageInformationContentSection = serviceDetails.storageInformationContentSection
        val storageInformation = if (storageInformationContentSection != null) {
            sectionMapper.storageInformation(storageInformationContentSection)
        } else {
            null
        }

        // TODO: move this order logic to the core module

        val shortDisplayFormat = serviceDetails.dpsDisplayFormat != null && serviceDetails.dpsDisplayFormat == DpsDisplayFormat.SHORT

        if (shortDisplayFormat) {
            return listOfNotNull(
                sectionMapper.getServiceDescription(serviceDetails, labels),
                sectionMapper.getProcessingCompany(serviceDetails, labels),
                sectionMapper.getDataPurposes(serviceDetails, labels),
                sectionMapper.getDataCollected(serviceDetails, labels),
                sectionMapper.getDataRecipients(serviceDetails, labels),
                sectionMapper.getHistory(serviceDetails, labels)
            )
        }

        return listOfNotNull(
            sectionMapper.getServiceDescription(serviceDetails, labels),
            sectionMapper.getProcessingCompany(serviceDetails, labels),
            sectionMapper.getDataPurposes(serviceDetails, labels),
            sectionMapper.getTechnologiesUsed(serviceDetails, labels),
            sectionMapper.getDataCollected(serviceDetails, labels),
            sectionMapper.getLegalBasis(serviceDetails, labels),
            sectionMapper.getProcessingLocation(serviceDetails, labels),
            sectionMapper.getRetentionPeriod(serviceDetails, labels),
            sectionMapper.getThirdPartyCountriesDistribution(serviceDetails, labels),
            sectionMapper.getDataRecipients(serviceDetails, labels),
            sectionMapper.getPrivacyPolicy(serviceDetails, labels),
            sectionMapper.getCookiePolicy(serviceDetails, labels),
            sectionMapper.getOptOutLink(serviceDetails, labels),
            sectionMapper.getDataProcessingAgreement(serviceDetails, labels),
            storageInformation,
            sectionMapper.getHistory(serviceDetails, labels)
        )
    }

    private fun contentSectionsFrom(serviceContentSection: List<PredefinedUIServiceContentSection>): List<UCContentSectionPM> {
        return serviceContentSection.map { contentSection ->
            when (val serviceContent = contentSection.content) {
                is PredefinedUISimpleServiceContent -> {
                    UCContentTextSectionPM(
                        title = contentSection.title,
                        description = serviceContent.content
                    )
                }

                is PredefinedUIHyperlinkServiceContent -> {
                    val url = serviceContent.url
                    UCContentTextSectionPM(
                        title = contentSection.title,
                        link = UCLinkPMLegacy(
                            label = url,
                            callback = sectionMapper.createUrlCallback(url)
                        )
                    )
                }

                is PredefinedUIStorageInformationServiceContent -> {
                    sectionMapper.storageInformation(contentSection)
                }
            }
        }
    }
}
