package com.usercentrics.sdk.ui.mappers

import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.ui.components.cards.UCCardHistoryEntryPM
import com.usercentrics.sdk.ui.components.cards.UCContentHistorySectionPM
import com.usercentrics.sdk.ui.components.cards.UCContentSectionPM
import com.usercentrics.sdk.ui.components.cards.UCContentTextSectionPM
import com.usercentrics.sdk.ui.components.links.UCLinkPMLegacy
import com.usercentrics.sdk.ui.extensions.filterNotBlank

internal class UCServiceSectionMapper(
    private val onOpenUrl: (String) -> Unit,
    private val onShowCookiesDialog: (PredefinedUIStorageInformationButtonInfo) -> Unit
) {
    fun getServiceDescription(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        return if (service.serviceDescription.isNotBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.descriptionTitle,
                description = service.serviceDescription
            )
        } else {
            null
        }
    }

    fun getProcessingCompany(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val processingCompanyBld = StringBuilder()
        val name = service.processingCompany?.name
        if (!name.isNullOrBlank()) {
            processingCompanyBld.append(name)
        }
        val address = service.processingCompany?.address
        if (!address.isNullOrBlank()) {
            if (processingCompanyBld.isNotEmpty()) {
                processingCompanyBld.append("\n")
            }
            processingCompanyBld.append(address)
        }
        val processingCompany = processingCompanyBld.toString()
        return if (processingCompany.isNotBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.processingCompanyTitle,
                description = processingCompany
            )
        } else {
            null
        }
    }

    fun getDataPurposes(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val dataPurposes = service.dataPurposes.filterNotBlank()
        return if (dataPurposes.isNotEmpty()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.dataPurposes.title,
                description = internationalizationLabels.service.dataPurposes.titleDescription,
                tags = dataPurposes
            )
        } else {
            null
        }
    }

    fun getTechnologiesUsed(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val technologiesUsed = service.technologiesUsed.filterNotBlank()
        return if (technologiesUsed.isNotEmpty()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.technologiesUsed.title,
                description = internationalizationLabels.service.technologiesUsed.titleDescription,
                tags = technologiesUsed
            )
        } else {
            null
        }
    }

    fun getDataCollected(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val dataCollected = service.dataCollected.filterNotBlank()
        return if (dataCollected.isNotEmpty()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.dataCollected.title,
                description = internationalizationLabels.service.dataCollected.titleDescription,
                tags = dataCollected
            )
        } else {
            null
        }
    }

    fun getLegalBasis(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val legalBasisList = service.legalBasis.filterNotBlank()
        return if (legalBasisList.isNotEmpty()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.legalBasis.title,
                description = internationalizationLabels.service.legalBasis.titleDescription,
                tags = legalBasisList
            )
        } else {
            null
        }
    }

    fun getProcessingLocation(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val processingLocation = service.dataDistribution?.processingLocation
        return if (!processingLocation.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.dataDistribution.processingLocationTitle,
                description = processingLocation
            )
        } else {
            null
        }
    }

    fun getRetentionPeriod(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        return if (service.retentionPeriodDescription.isNotBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.retentionPeriodTitle,
                description = service.retentionPeriodDescription
            )
        } else {
            null
        }
    }

    fun getThirdPartyCountriesDistribution(
        service: PredefinedUIServiceDetails,
        labels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val thirdPartyCountries = service.dataDistribution?.thirdPartyCountries
        return if (!thirdPartyCountries.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = labels.service.dataDistribution.thirdPartyCountriesTitle,
                description = labels.service.dataDistribution.thirdPartyCountriesDescription,
                tags = listOf(thirdPartyCountries),
            )
        } else {
            null
        }
    }

    fun getDataRecipients(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val dataRecipients = service.dataRecipients.filterNotBlank()
        return if (dataRecipients.isNotEmpty()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.dataRecipientsTitle,
                tags = dataRecipients
            )
        } else {
            null
        }
    }

    fun getPrivacyPolicy(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val privacyPolicy = service.urls?.privacyPolicy
        return if (!privacyPolicy.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.urls.privacyPolicyTitle,
                link = UCLinkPMLegacy(
                    label = privacyPolicy,
                    callback = createUrlCallback(privacyPolicy),
                )
            )
        } else {
            null
        }
    }

    fun getCookiePolicy(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val cookiePolicy = service.urls?.cookiePolicy
        return if (!cookiePolicy.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.urls.cookiePolicyTitle,
                link = UCLinkPMLegacy(
                    label = cookiePolicy,
                    callback = createUrlCallback(cookiePolicy)
                )
            )
        } else {
            null
        }
    }

    fun getOptOutLink(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val optOut = service.urls?.optOut
        return if (!optOut.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.urls.optOutTitle,
                link = UCLinkPMLegacy(
                    label = optOut,
                    callback = createUrlCallback(optOut)
                )
            )
        } else {
            null
        }
    }

    fun getDataProcessingAgreement(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentTextSectionPM? {
        val dataProcessingAgreement = service.urls?.dataProcessingAgreement
        return if (!dataProcessingAgreement.isNullOrBlank()) {
            UCContentTextSectionPM(
                title = internationalizationLabels.service.urls.dataProcessingAgreementTitle,
                link = UCLinkPMLegacy(
                    label = dataProcessingAgreement,
                    callback = createUrlCallback(dataProcessingAgreement)
                )
            )
        } else {
            null
        }
    }

    fun getHistory(
        service: PredefinedUIServiceDetails,
        internationalizationLabels: PredefinedUILabels
    ): UCContentHistorySectionPM? {
        val history = service.consent?.history
        return if (!history.isNullOrEmpty()) {
            UCContentHistorySectionPM(
                title = internationalizationLabels.service.history.title,
                history = history.reversed().map {
                    UCCardHistoryEntryPM(
                        status = it.status,
                        date = it.formattedDate,
                        decisionText = it.decisionText
                    )
                },
                decisionLabel = internationalizationLabels.general.decision,
                dateLabel = internationalizationLabels.general.date
            )
        } else {
            null
        }
    }

    fun createUrlCallback(url: String): () -> Unit {
        return {
            onOpenUrl(url)
        }
    }

    fun storageInformation(contentSection: PredefinedUIServiceContentSection): UCContentSectionPM {
        val content = contentSection.content as PredefinedUIStorageInformationServiceContent
        val storageInformationButton = content.button
        val storageLink = if (storageInformationButton != null) {
            UCLinkPMLegacy(
                label = storageInformationButton.label,
                callback = {
                    onShowCookiesDialog(storageInformationButton)
                }
            )
        } else {
            null
        }

        return UCContentTextSectionPM(
            title = contentSection.title,
            description = content.content,
            link = storageLink
        )
    }
}
