package com.usercentrics.sdk.ui.secondLayer

import android.content.Context
import com.usercentrics.sdk.*
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.predefinedUI.PredefinedUIConsentManager
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.ui.PredefinedUIDependencyManager
import com.usercentrics.sdk.ui.banner.SecondLayerInitialState
import com.usercentrics.sdk.ui.banner.UCBannerCoordinator
import com.usercentrics.sdk.ui.components.UCButtonType
import com.usercentrics.sdk.ui.components.cards.UCControllerIdPM
import com.usercentrics.sdk.ui.components.cookie.UCCookiesDialog
import com.usercentrics.sdk.ui.extensions.copyToClipboard
import com.usercentrics.sdk.ui.extensions.openUrl
import com.usercentrics.sdk.ui.mappers.UCCategoryMapper
import com.usercentrics.sdk.ui.mappers.UCCategoryMapperImpl
import com.usercentrics.sdk.ui.mappers.UCServiceMapper
import com.usercentrics.sdk.ui.mappers.UCServiceMapperImpl
import com.usercentrics.sdk.ui.secondLayer.component.footer.UCSecondLayerFooterViewModel
import com.usercentrics.sdk.ui.secondLayer.component.footer.UCSecondLayerFooterViewModelImpl
import com.usercentrics.sdk.ui.secondLayer.component.header.UCSecondLayerHeaderViewModel
import com.usercentrics.sdk.ui.secondLayer.component.header.UCSecondLayerHeaderViewModelImpl
import com.usercentrics.sdk.ui.theme.UCThemeData
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleGroup
import com.usercentrics.sdk.ui.toggle.PredefinedUIToggleMediator

internal interface UCBaseLayerViewModel {

    val statusBarColor: Int?
    fun onButtonClick(type: UCButtonType)
}

internal interface UCSecondLayerViewModel : UCBaseLayerViewModel {

    val labels: PredefinedUILabels
    val showCloseButton: Boolean
    val customLogo: UsercentricsImage?
    val theme: UCThemeData
    val optOutToggleValue: Boolean

    fun bind(callback: SecondLayerViewModelBindCallback)
    fun onClosePressed()
    fun onSelectLanguage(selectedLanguage: String)
    fun onOptOutSwitchChanged(state: Boolean)
    fun onLinkClick(link: PredefinedUILink)
}

internal typealias SecondLayerViewModelBindCallback = (content: UCLayerContentPM, header: UCSecondLayerHeaderViewModel, footer: UCSecondLayerFooterViewModel) -> Unit

internal class UCSecondLayerViewModelImpl(
    private val context: Context,
    private val toggleMediator: PredefinedUIToggleMediator,
    private val consentManager: PredefinedUIConsentManager,
    private val viewHandlers: PredefinedUIViewHandlers,
    private var layerSettings: UCUISecondLayerSettings,
    private val controllerId: String,
    private val settings: SecondLayerStyleSettings?,
    private val initialState: SecondLayerInitialState?,
    override val customLogo: UsercentricsImage?,
    override var labels: PredefinedUILabels,
    override val theme: UCThemeData,
    private val landscapeMode: Boolean,
    private val coordinator: UCBannerCoordinator,
    private val linksSettings: LegalLinksSettings,
    override val statusBarColor: Int?
) : UCSecondLayerViewModel {

    companion object {
        private const val defaultInitialTabIndex = 0
        private const val defaultShowCloseButton = false
    }

    private val categoryMapper: UCCategoryMapper = UCCategoryMapperImpl()
    private val serviceMapper: UCServiceMapper = UCServiceMapperImpl(context::openUrl, ::showCookiesDialog)
    private var bindCallback: SecondLayerViewModelBindCallback? = null

    private var _optOutToggleValue: Boolean = initialState?.ccpaToggleValue ?: layerSettings.footerSettings.optOutToggleInitialValue
    override val optOutToggleValue: Boolean
        get() = _optOutToggleValue

    override val showCloseButton: Boolean
        get() = settings?.showCloseButton ?: defaultShowCloseButton

    override fun bind(callback: SecondLayerViewModelBindCallback) {
        this.bindCallback = callback.also {
            it.bindData()
        }
    }

    override fun onSelectLanguage(selectedLanguage: String) {
        viewHandlers.updateLanguage(
            selectedLanguage,
            /* onSuccess */ {
                layerSettings = it.settings.secondLayerV2
                labels = it.settings.internationalizationLabels
                bindCallback?.bindData()
            },
            /*onFailure */ {
                // Do nothing
            }
        )
    }

    override fun onOptOutSwitchChanged(state: Boolean) {
        _optOutToggleValue = state
    }

    override fun onButtonClick(type: UCButtonType) {
        when (type) {
            UCButtonType.ACCEPT_ALL -> onAcceptAllSettingsClick()
            UCButtonType.DENY_ALL -> onDenyAllSettingsClick()
            UCButtonType.OK -> onOkSettingsClick()
            UCButtonType.SAVE -> onSaveSettingsClick()
            UCButtonType.MORE -> assert(false) { "MORE button does not apply to the second layer" }
        }
    }

    override fun onLinkClick(link: PredefinedUILink) {
        when (link.linkType) {
            PredefinedUILinkType.URL -> onHyperlinkClick(link)
            PredefinedUILinkType.MANAGE_SETTINGS -> Unit // this link does not apply to the second layer
            PredefinedUILinkType.VENDOR_LIST -> Unit // this link does not apply to the second layer
        }
    }

    private fun onHyperlinkClick(link: PredefinedUILink) {
        coordinator.navigateToUrl(link.url ?: "")
        trackAnalyticsEvent(link.eventType)
    }

    private fun onSaveSettingsClick() {
        val feedback = consentManager.save(
            TCFDecisionUILayer.SECOND_LAYER,
            toggleMediator.getUserDecisions(),
        )
        coordinator.finishCMP(feedback.toUserResponse())
    }

    private fun onOkSettingsClick() {
        val feedback = if (optOutToggleValue) {
            consentManager.denyAll(TCFDecisionUILayer.SECOND_LAYER)
        } else {
            consentManager.acceptAll(TCFDecisionUILayer.SECOND_LAYER)
        }
        coordinator.finishCMP(feedback.toUserResponse())
    }

    private fun onAcceptAllSettingsClick() {
        val feedback = consentManager.acceptAll(TCFDecisionUILayer.SECOND_LAYER)
        coordinator.finishCMP(feedback.toUserResponse())
    }

    private fun onDenyAllSettingsClick() {
        val feedback = consentManager.denyAll(TCFDecisionUILayer.SECOND_LAYER)
        coordinator.finishCMP(feedback.toUserResponse())
    }

    override fun onClosePressed() {
        val feedback = consentManager.close()
        coordinator.finishCMP(feedback.toUserResponse())
    }

    // TODO move to the Banner coordinator
    private fun showCookiesDialog(storageInformationButton: PredefinedUIStorageInformationButtonInfo) {
        UCCookiesDialog(theme, storageInformationButton).show(context)
    }

    private fun onCopyControllerId() {
        context.copyToClipboard(
            controllerId,
            labels.general.controllerId
        )
    }

    private fun buildContent(): UCLayerContentPM {
        val tabs = layerSettings.contentSettings.map { tab ->
            when (val tabContent = tab.content) {
                is PredefinedUIServicesContent -> UCLayerTabPM(tab.title, buildServicesContent(tabContent))
                is PredefinedUICategoriesContent -> UCLayerTabPM(tab.title, buildCategoriesContent(tabContent))
            }
        }
        val initialTabIndex = initialState?.tabIndex ?: defaultInitialTabIndex
        return UCLayerContentPM(initialTabIndex, tabs)
    }

    private fun buildServicesContent(tabContent: PredefinedUIServicesContent): List<UCCardsContentPM> {
        return tabContent.cardUISections.map { cardUISection ->
            UCCardsContentPM(
                title = cardUISection.title,
                cards = cardUISection.cards.map { service ->
                    val group = service.createToggleGroup()
                    serviceMapper.map(service, group, toggleMediator, labels)
                },
                controllerId = buildControllerID(cardUISection)
            )
        }
    }

    private fun buildControllerID(cardUISection: PredefinedUICardUISection): UCControllerIdPM? {
        val controllerID = cardUISection.controllerID ?: return null
        return UCControllerIdPM(
            label = controllerID.label,
            value = controllerID.value,
            ariaLabel = labels.ariaLabels.copyControllerId ?: "",
            onCopyControllerId = ::onCopyControllerId
        )
    }

    private fun buildCategoriesContent(tabContent: PredefinedUICategoriesContent): List<UCCardsContentPM> {
        return tabContent.cardUISections.map { cardUISection ->
            UCCardsContentPM(
                title = cardUISection.title,
                cards = cardUISection.cards.map { category ->
                    val group = category.createToggleGroup()
                    categoryMapper.map(category, group, toggleMediator)
                },
                controllerId = buildControllerID(cardUISection)
            )
        }
    }

    private fun PredefinedUICardUI.createToggleGroup(): PredefinedUIToggleGroup? {
        return toggleMediator.getGroupLegacy(this)
    }

    private fun SecondLayerViewModelBindCallback.bindData() {
        invoke(
            buildContent(),
            UCSecondLayerHeaderViewModelImpl(
                settings = layerSettings.headerSettings,
                linksSettings = linksSettings,
                parentViewModel = this@UCSecondLayerViewModelImpl,
            ),
            UCSecondLayerFooterViewModelImpl(
                settings = layerSettings.footerSettings,
                buttonLayout = settings?.buttonLayout,
                buttonLabels = labels.firstLayerButtonLabels,
                theme = theme,
                landscapeMode = landscapeMode,
                parentViewModel = this@UCSecondLayerViewModelImpl
            ),
        ).also {
            toggleMediator.bootLegacy()
        }
    }

    private fun trackAnalyticsEvent(event: UsercentricsAnalyticsEventType) {
        PredefinedUIDependencyManager.analyticsManager.track(event)
    }
}
