package com.usercentrics.sdk.ui.secondLayer.component.adapters

import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import androidx.viewpager.widget.PagerAdapter
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.components.cards.UCCardComponent
import com.usercentrics.sdk.ui.secondLayer.UCLayerTabPM
import com.usercentrics.sdk.ui.theme.UCThemeData
internal const val PAGE_TAG = "page_"

internal class UCSecondLayerTabsPagerAdapter(
    private val theme: UCThemeData,
    private val navigateToTab: (position: Int) -> Unit,
    private val collapseHeader: () -> Unit
) : PagerAdapter() {

    var contentTabs: List<UCLayerTabPM> = emptyList()
        set(value) {
            field = value
            adapterToTabPosition.forEach { (adapter, position) ->
                val tabContent = value.getOrNull(position)?.content ?: return@forEach
                adapter.cardComponents = UCCardComponent.from(tabContent)
            }
            notifyDataSetChanged()
        }

    private val rvToAdapter = mutableMapOf<RecyclerView, UCSecondLayerCardsAdapter>()
    private val adapterToTabPosition = mutableMapOf<UCSecondLayerCardsAdapter, Int>()

    override fun instantiateItem(container: ViewGroup, position: Int): Any {
        val cardVerticalMargin = container.resources.getDimension(R.dimen.ucCardVerticalMargin).toInt()
        return RecyclerView(container.context).apply {
            contentDescription = "Tab list $position"
            tag = PAGE_TAG + position
            id = when (position) {
                0 -> R.id.ucHeaderFirstTabRecyclerView
                1 -> R.id.ucHeaderSecondTabRecyclerView
                else -> View.NO_ID
            }
            layoutManager = LinearLayoutManager(container.context)
            adapter = UCSecondLayerCardsAdapter(
                theme = theme,
                onMoreInfo = this@UCSecondLayerTabsPagerAdapter::navigateToCard,
            ) {
                collapseHeader()
                centerCardBy(it, this, cardVerticalMargin)
            }.also {
                rvToAdapter[this] = it
                adapterToTabPosition[it] = position
                val tabContent = contentTabs.getOrNull(position)?.content ?: return@also
                it.cardComponents = UCCardComponent.from(tabContent)
            }
            container.addView(this)
            theme.colorPalette.layerBackgroundSecondaryColor?.let { setBackgroundColor(it) }
            // Request layout the next frame right after to ensure that the layout size is correct
            post { requestLayout() }
        }
    }

    private fun navigateToCard(cardId: String) {
        var cardPosition = -1
        val entry = rvToAdapter.entries.firstOrNull {
            cardPosition = it.value.cardPosition(cardId)
            cardPosition > -1
        } ?: return

        val (recyclerView, cardsAdapter) = entry
        val tabPosition = adapterToTabPosition[cardsAdapter] ?: return

        navigateToTab(tabPosition)
        collapseHeader()

        cardsAdapter.collapseAll(false)
        cardsAdapter.setExpanded(cardPosition, false)

        cardsAdapter.notifyDataSetChanged()
        centerOnPosition(cardPosition, recyclerView, cardsAdapter)
    }

    private fun centerOnPosition(
        position: Int,
        recyclerView: RecyclerView,
        adapter: RecyclerView.Adapter<RecyclerView.ViewHolder>
    ) {
        val lastItemPosition = adapter.itemCount - 1
        recyclerView.scrollToPosition(lastItemPosition)
        recyclerView.post {
            recyclerView.scrollToPosition(position)
        }
    }

    private fun centerCardBy(cardY: Int, recyclerView: RecyclerView, cardVerticalMargin: Int) {
        val rvLocation = intArrayOf(0, 0)
        recyclerView.getLocationOnScreen(rvLocation)
        val rvY = rvLocation[1]
        val dy = cardY - rvY - cardVerticalMargin
        recyclerView.smoothScrollBy(0, dy)
    }

    override fun getCount(): Int = contentTabs.size

    override fun isViewFromObject(view: View, obj: Any): Boolean {
        return view == obj
    }

    override fun destroyItem(container: ViewGroup, position: Int, obj: Any) {
        container.removeView(obj as View)
        val adapter = rvToAdapter.remove(obj as? RecyclerView) ?: return
        adapterToTabPosition.remove(adapter)
    }

    override fun getPageTitle(position: Int): CharSequence {
        return contentTabs.getOrNull(position)?.title ?: ""
    }
}
