package com.usercentrics.sdk.ui.secondLayer.component.header

import android.content.Context
import android.os.Build
import android.text.method.LinkMovementMethod
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnClickListener
import android.view.ViewGroup
import android.view.ViewStub
import android.widget.ProgressBar
import androidx.annotation.AttrRes
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.viewpager.widget.ViewPager
import com.google.android.material.tabs.TabLayout
import com.usercentrics.sdk.models.settings.PredefinedUILink
import com.usercentrics.sdk.ui.R
import com.usercentrics.sdk.ui.accessibility.accessibleTouchTarget
import com.usercentrics.sdk.ui.accessibility.createAccessibilityLinkOrButton
import com.usercentrics.sdk.ui.components.UCImageView
import com.usercentrics.sdk.ui.components.UCTextView
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getBackButtonIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.getCloseIcon
import com.usercentrics.sdk.ui.components.drawable.ThemedDrawable.styleIcon
import com.usercentrics.sdk.ui.components.links.LinksViewBuilder
import com.usercentrics.sdk.ui.components.style
import com.usercentrics.sdk.ui.extensions.dpToPx
import com.usercentrics.sdk.ui.extensions.setVerticalPadding
import com.usercentrics.sdk.ui.popup.LanguagePopup
import com.usercentrics.sdk.ui.theme.UCThemeData
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition

internal class UCSecondLayerHeader : ConstraintLayout {

    private val stubView by lazy { findViewById<ViewStub>(R.id.ucHeaderStub) }
    private lateinit var inflatedStubView: View

    private val ucHeaderLogo by lazy { inflatedStubView.findViewById<UCImageView>(R.id.ucHeaderLogo) }
    private val ucHeaderLanguageIcon by lazy { inflatedStubView.findViewById<UCImageView>(R.id.ucHeaderLanguageIcon) }
    private val ucHeaderLanguageLoading by lazy { inflatedStubView.findViewById<ProgressBar>(R.id.ucHeaderLanguageLoading) }
    private val ucHeaderBackButton by lazy { inflatedStubView.findViewById<UCImageView>(R.id.ucHeaderBackButton) }

    private val ucHeaderCloseButton by lazy { findViewById<UCImageView>(R.id.ucHeaderCloseButton) }
    private val ucHeaderTitle by lazy { findViewById<UCTextView>(R.id.ucHeaderTitle) }
    private val ucHeaderDescription by lazy { findViewById<UCTextView>(R.id.ucHeaderDescription) }
    private val ucHeaderLinks by lazy { findViewById<ViewGroup>(R.id.ucHeaderLinks) }
    private val ucHeaderTabLayout by lazy { findViewById<TabLayout>(R.id.ucHeaderTabLayout) }

    private val ucHeaderContentDivider by lazy { findViewById<View>(R.id.ucHeaderContentDivider) }

    private lateinit var viewModel: UCSecondLayerHeaderViewModel

    private var alreadySetHeaderTabPosition: Boolean = false

    private val linksVerticalPadding = 2.dpToPx(context)
    private val linksHorizontalSpacing = resources.getDimensionPixelOffset(R.dimen.ucFirstLayerLinksHorizontalSpacing)

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        initLayout(context)
    }

    private fun initLayout(context: Context) {
        LayoutInflater.from(context).inflate(R.layout.uc_header, this)
        ucHeaderTabLayout.isTabIndicatorFullWidth = false
        ucHeaderDescription.movementMethod = LinkMovementMethod.getInstance()
    }

    fun style(theme: UCThemeData) {
        ucHeaderTitle.styleTitle(theme)
        ucHeaderDescription.styleBody(theme)
        ucHeaderTabLayout.style(theme)
        ucHeaderContentDivider.setBackgroundColor(theme.colorPalette.tabsBorderColor)
        theme.colorPalette.layerBackgroundColor?.let { setBackgroundColor(it) }
        ucHeaderTabLayout.clearOnTabSelectedListeners()
        ucHeaderTabLayout.addOnTabSelectedListener(StyleTabListener(theme))
    }

    fun bind(theme: UCThemeData, model: UCSecondLayerHeaderViewModel) {
        this.viewModel = model

        bindHeaderTabPosition(theme)
        bindLogoAndNavButtons()
        bindLanguage(theme)
        bindDescription()
        bindLinks(theme)

        ucHeaderTitle.text = model.title

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            ucHeaderTitle.isAccessibilityHeading = true
        } else {
            ucHeaderTitle.importantForAccessibility = View.IMPORTANT_FOR_ACCESSIBILITY_YES
        }

        ucHeaderLanguageIcon.isFocusable = true
        ucHeaderCloseButton.isFocusable = true
        ucHeaderTitle.isFocusable = true
        ucHeaderDescription.isFocusable = true
        ucHeaderLanguageIcon.requestFocus()
    }

    fun bindTabs(theme: UCThemeData, viewPager: ViewPager, tabNames: List<String>, showTabs: Boolean) {
        ucHeaderTabLayout.setupWithViewPager(viewPager)

        if (showTabs) {
            setupHeaderTabs(theme, tabNames, viewPager.currentItem)
            ucHeaderTabLayout.visibility = VISIBLE
            (ucHeaderContentDivider.layoutParams as LayoutParams).topMargin = 0
        } else {
            ucHeaderTabLayout.visibility = View.GONE
            (ucHeaderContentDivider.layoutParams as LayoutParams).topMargin = 8.dpToPx(context)
        }
    }

    private fun bindHeaderTabPosition(theme: UCThemeData) {
        if (alreadySetHeaderTabPosition) {
            return
        }

        setupHeaderItemsPosition()
        setupLanguage(theme)
        setupBackButton(theme)
        setupCloseButton(theme)

        alreadySetHeaderTabPosition = true
    }

    private fun setupHeaderItemsPosition() {
        stubView.layoutResource = when (viewModel.logoPosition) {
            FirstLayerLogoPosition.LEFT -> R.layout.uc_header_items_left
            FirstLayerLogoPosition.CENTER -> R.layout.uc_header_items_center
            FirstLayerLogoPosition.RIGHT -> R.layout.uc_header_items_right
        }

        inflatedStubView = stubView.inflate()
    }

    private fun setupLanguage(theme: UCThemeData) {
        ucHeaderLanguageIcon.styleIcon(theme)
        ucHeaderLanguageLoading.style(theme)
        ucHeaderLanguageIcon.accessibleTouchTarget()
    }

    private fun setupBackButton(theme: UCThemeData) {
        val icon = context.getBackButtonIcon()?.also {
            it.styleIcon(theme)
        }
        ucHeaderBackButton.setImageDrawable(icon)
        ucHeaderBackButton.accessibleTouchTarget()
    }

    private fun setupCloseButton(theme: UCThemeData) {
        val icon = context.getCloseIcon()?.also {
            it.styleIcon(theme)
        }

        ucHeaderCloseButton.apply {
            setImageDrawable(icon)
            setOnClickListener { viewModel.onCloseButton() }
        }
    }

    private fun bindLogoAndNavButtons() {
        setLogoMode()
        showCloseButtonIfNeeded()
    }

    private fun setLogoMode() {
        val logoImage = viewModel.logoImage

        ucHeaderBackButton.visibility = View.GONE
        ucHeaderBackButton.setOnClickListener(null)
        ucHeaderLogo.visibility = View.VISIBLE

        val showLogoPlaceholder = logoImage == null
        if (showLogoPlaceholder) {
            return
        }

        ucHeaderLogo.apply {
            setImage(logoImage!!)
            contentDescription = viewModel.ariaLabels.logoAltTag
        }
    }

    private fun showCloseButtonIfNeeded() {
        val closeButtonVisibility = if (viewModel.showCloseButton) {
            View.VISIBLE
        } else {
            View.GONE
        }

        ucHeaderCloseButton.apply {
            visibility = closeButtonVisibility
            contentDescription = viewModel.ariaLabels.closeButton
            accessibleTouchTarget()
        }
    }

    private fun setupHeaderTabs(theme: UCThemeData, tabNames: List<String>, currentSelectedIndex: Int) {
        tabNames.forEachIndexed { index, tabName ->
            ucHeaderTabLayout?.getTabAt(index)?.apply {
                val tabText = UCTextView(context).apply {
                    text = tabName
                    id = when (index) {
                        0 -> R.id.ucHeaderFirstTabView
                        1 -> R.id.ucHeaderSecondTabView
                        else -> View.NO_ID
                    }
                }
                tabText.styleTab(theme)
                customView = tabText

                val isSelected = currentSelectedIndex == index
                if (isSelected) {
                    tabText.styleSelectedTab(theme)
                } else {
                    tabText.styleUnselectedTab(theme)
                }
            }
        }
    }

    private inner class StyleTabListener(
        private val theme: UCThemeData
    ) : TabLayout.OnTabSelectedListener {
        override fun onTabSelected(tab: TabLayout.Tab?) {
            val customView = tab?.customView as? UCTextView ?: return
            customView.styleSelectedTab(theme)
        }

        override fun onTabUnselected(tab: TabLayout.Tab?) {
            val customView = tab?.customView as? UCTextView ?: return
            customView.styleUnselectedTab(theme)
        }

        override fun onTabReselected(tab: TabLayout.Tab?) {
        }
    }

    private fun bindLinks(theme: UCThemeData) {
        ucHeaderLinks.removeAllViews()

        val legalLinks = viewModel.legalLinks ?: emptyList()
        if (legalLinks.isEmpty()) {
            ucHeaderLinks.visibility = View.GONE
            return
        }

        val links = legalLinks.map { link -> createLinkView(link, theme) }

        val container = LinksViewBuilder.createLinksLayout(context, links, linksHorizontalSpacing)
        ucHeaderLinks.addView(container)
    }

    private fun createLinkView(link: PredefinedUILink, theme: UCThemeData): UCTextView {
        return UCTextView(context).apply {
            text = link.label
            setVerticalPadding(linksVerticalPadding)
            styleSmall(theme, isBold = true, isLink = true)
            setOnClickListener { viewModel.onLinkClick(link) }
            createAccessibilityLinkOrButton()
        }
    }

    private fun bindLanguage(theme: UCThemeData) {
        val languageSettings = viewModel.language
        ucHeaderLanguageLoading.visibility = View.GONE

        val languageIconVisibility = if (languageSettings == null) {
            View.INVISIBLE
        } else {
            View.VISIBLE
        }

        ucHeaderLanguageIcon.apply {
            visibility = languageIconVisibility
            contentDescription = viewModel.ariaLabels.languageSelector

            setOnClickListener(showLanguagePopupMenu(theme))
        }
    }

    private fun showLanguagePopupMenu(theme: UCThemeData): OnClickListener {
        return OnClickListener { targetView ->
            val languageSettings = viewModel.language ?: return@OnClickListener

            val languages = languageSettings.available
            if (languages.isEmpty()) {
                return@OnClickListener
            }

            val selectedIsoCode = languageSettings.selected.isoCode
            LanguagePopup(context, theme)
                .setLanguageSelectedListener(::onLanguageSelected)
                .show(targetView, languages, selectedIsoCode)
        }
    }

    private fun onLanguageSelected(selectedIsoCode: String) {
        val languageSettings = viewModel.language ?: return

        val canChangeLanguage = selectedIsoCode != languageSettings.selected.isoCode
        if (!canChangeLanguage) {
            return
        }

        ucHeaderLanguageIcon.visibility = View.INVISIBLE
        ucHeaderLanguageLoading.visibility = View.VISIBLE

        viewModel.onSelectLanguage(selectedIsoCode)
    }

    private fun bindDescription() {
        val contentDescription = viewModel.contentDescription
        ucHeaderDescription.setHtmlText(
            htmlText = contentDescription, predefinedUILinkHandler = viewModel::onPredefinedUIHtmlLinkClick
        )
    }
}
