package com.usercentrics.sdk.ui.toggle

import com.usercentrics.sdk.models.settings.*

// TODO: This will become a common view model when all the legacy stuff is migrated.
// This is because the legacy approach uses the card model to identify toggles
// and switch settings for others, that makes almost impossible to refactor this in a simple way.
// They idea is to do the migration once everything relies on the toggle settings.
// Long term plan:
// ToggleMediator -> ToggleViewModel
// getGroup -> bindState/onStateChange
interface PredefinedUIToggleMediator {
    fun bootLegacy()
    fun tearDown()

    fun getGroup(settings: PredefinedUIToggleSettings): PredefinedUIToggleGroup
    fun getGroupLegacy(cardUI: PredefinedUICardUI): PredefinedUIToggleGroup?
    fun getServiceGroupLegacy(id: String, switchSettings: PredefinedUISwitchSettingsUI): PredefinedUIToggleGroup

    fun getUserDecisions(): List<PredefinedUIDecision>
}

/**
 * ToggleMediator implementation base on the domain concept of 'Category' and 'Service'.
 * A category is an entity with a state dependents on the state of the services related with it.
 * This is a 1:N relationship. The mediation logic for this relationship is:
 * - The category state is True if any related service has its state to True.
 * - The category state is False if all the related services are false.
 * - If the category state change to True/False, all the related services will change to the same state.
 */
class PredefinedUIToggleMediatorImpl : PredefinedUIToggleMediator {

    private val categoryToServices: MutableMap<String, List<String>> = mutableMapOf()

    // <ID, <ConsentID, Toggle>>
    // Each ID could have several ConsentID, in other words, each ID could have several toggles.
    private val categoryGroups: MutableMap<String, MutableMap<String, PredefinedUIToggleGroup>> = mutableMapOf()
    private val servicesGroups: MutableMap<String, MutableMap<String, PredefinedUIToggleGroup>> = mutableMapOf()

    override fun tearDown() {
        categoryToServices.clear()
        for (entry in categoryGroups.values) {
            for (group in entry.values) {
                group.dispose()
            }
        }
        for (entry in servicesGroups.values) {
            for (group in entry.values) {
                group.dispose()
            }
        }
        categoryGroups.clear()
        servicesGroups.clear()
    }

    override fun getGroup(settings: PredefinedUIToggleSettings): PredefinedUIToggleGroup {
        return if (settings.isCategory()) {
            setCategoryServices(settings.id, serviceIds = settings.dependentsIds)
            categoryGroups.getToggleGroup(settings)
        } else {
            servicesGroups.getToggleGroup(settings)
        }
    }

    private fun PredefinedUIToggleSettings.isCategory(): Boolean {
        return dependentsIds.isNotEmpty()
    }

    private fun MutableMap<String, MutableMap<String, PredefinedUIToggleGroup>>.getToggleGroup(settings: PredefinedUIToggleSettings): PredefinedUIToggleGroup {
        val currentValue = this[settings.id] ?: run {
            val group = createGroup(settings)
            this[settings.id] = mutableMapOf(settings.consentId to group)
            return group
        }
        return currentValue[settings.consentId] ?: run {
            val group = createGroup(settings)
            currentValue[settings.consentId] = group
            return group
        }
    }

    private fun createGroup(settings: PredefinedUIToggleSettings): PredefinedUIToggleGroup {
        return PredefinedUIToggleGroupImpl(currentState = settings.currentValue).apply {
            if (settings.isCategory()) {
                setListener { status ->
                    handleToggledCategory(categorySlug = settings.id, switchId = settings.consentId, isChecked = status)
                }
            } else {
                setListener { status ->
                    handleToggledService(serviceId = settings.id, switchId = settings.consentId, isChecked = status)
                }
            }
        }
    }

    private fun setCategoryServices(categorySlug: String, serviceIds: List<String>) {
        categoryToServices[categorySlug] = serviceIds
    }

    override fun getUserDecisions(): List<PredefinedUIDecision> {
        return servicesGroups.map { entry ->
            val serviceId = entry.key
            val consentMap = entry.value
            PredefinedUIDecision(
                serviceId = serviceId,
                values = consentMap.map {
                    it.key to it.value.currentState
                }.toMap()
            )
        }
    }

    private fun handleToggledCategory(categorySlug: String, switchId: String, isChecked: Boolean) {
        for (serviceId in categoryToServices[categorySlug] ?: return) {
            updateServiceState(serviceId, switchId, isChecked)
        }
    }

    private fun handleToggledService(serviceId: String, switchId: String, isChecked: Boolean) {
        updateServiceState(serviceId, switchId, isChecked)

        val hasCategorySlug = categoryToServices.entries.firstOrNull { it.value.contains(serviceId) }?.key
        if (hasCategorySlug != null) {
            handleCategoryToggledFromService(hasCategorySlug)
        }
    }

    private fun updateServiceState(serviceId: String, switchId: String, isChecked: Boolean) {
        servicesGroups[serviceId]?.get(switchId)?.currentState = isChecked
    }

    private fun handleCategoryToggledFromService(categorySlug: String?) {
        val categoryToggle = categoryGroups[categorySlug]
        val anyServiceIsChecked = categoryToServices[categorySlug]?.any { nestedServiceId ->
            servicesGroups[nestedServiceId]?.values?.any { group ->
                group.currentState
            } ?: false
        }
        for (group in categoryToggle?.values ?: return) {
            group.currentState = anyServiceIsChecked == true
        }
    }

    //region Legacy Implementation

    // Main differences:
    // - Need to invoke 'boot' each time that all the groups and mediation rules are created.
    // - The Consent ID concept is not clearly defined. It requires a combination of Card settings and several Switch settings models.

    override fun bootLegacy() {
        for (categoryEntry in categoryGroups.entries) {
            for (groupEntry in categoryEntry.value.entries) {
                groupEntry.value.setListener { status ->
                    handleToggledCategory(categorySlug = categoryEntry.key, switchId = groupEntry.key, isChecked = status)
                }
            }
        }
        for (serviceEntry in servicesGroups.entries) {
            for (groupEntry in serviceEntry.value.entries) {
                groupEntry.value.setListener { status ->
                    handleToggledService(serviceId = serviceEntry.key, switchId = groupEntry.key, isChecked = status)
                }
            }
        }
    }

    override fun getGroupLegacy(cardUI: PredefinedUICardUI): PredefinedUIToggleGroup? {
        val switchSettings = cardUI.mainSwitchSettings ?: return null
        val id = cardUI.id

        val dependantSwitches = cardUI.dependantSwitchSettings
        if (dependantSwitches.isNullOrEmpty()) {
            return this.getServiceGroupLegacy(id, switchSettings)
        }
        return buildSwitchWithDependantsLegacy(id, dependantSwitches, switchSettings)
    }

    private fun getCategoryGroupLegacy(id: String, switchSettings: PredefinedUISwitchSettingsUI): PredefinedUIToggleGroup {
        return categoryGroups.getToggleGroupLegacy(id, switchSettings)
    }

    override fun getServiceGroupLegacy(id: String, switchSettings: PredefinedUISwitchSettingsUI): PredefinedUIToggleGroup {
        return servicesGroups.getToggleGroupLegacy(id, switchSettings)
    }

    private fun MutableMap<String, MutableMap<String, PredefinedUIToggleGroup>>.getToggleGroupLegacy(
        id: String,
        switchSettings: PredefinedUISwitchSettingsUI
    ): PredefinedUIToggleGroup {
        val currentValue = this[id] ?: run {
            val group = createGroupLegacy(switchSettings)
            this[id] = mutableMapOf(switchSettings.id to group)
            return group
        }

        return currentValue[switchSettings.id] ?: run {
            val group = createGroupLegacy(switchSettings)
            currentValue[switchSettings.id] = group
            return group
        }
    }

    private fun buildSwitchWithDependantsLegacy(
        id: String,
        dependantSwitches: List<PredefinedUIDependantSwitchSettings>,
        switchSettings: PredefinedUISwitchSettingsUI
    ): PredefinedUIToggleGroup {
        val servicesIds = mutableListOf<String>()
        for (switch in dependantSwitches) {
            getServiceGroupLegacy(switch.id, switch.switchSettings)
            servicesIds.add(switch.id)
        }
        setCategoryServices(id, serviceIds = servicesIds)
        return getCategoryGroupLegacy(id, switchSettings)
    }

    private fun createGroupLegacy(switchSettings: PredefinedUISwitchSettingsUI): PredefinedUIToggleGroup {
        return PredefinedUIToggleGroupImpl(currentState = switchSettings.currentValue)
    }
    //endregion
}
