package com.usercentrics.ccpa

internal class CcpaApi(
    private val storage: CCPAStorage,
    private val debug: (String) -> Unit = {}
) {

    companion object {
        const val privacyStringStorageKey = "IABUSPrivacy_String"
        const val supportedApiVersion = 1
    }

    /**
     * Get the CCPA privacy data from Storage.
     */
    fun getPrivacyDataAsString(apiVersion: Int): String {
        checkApiVersion(apiVersion)
        return getStringFromStorage()
    }

    /**
     * Get the CCPA privacy data previously selected or the initial value.
     */
    fun getPrivacyData(apiVersion: Int): CCPAData {
        val ccpaString = getPrivacyDataAsString(apiVersion)
        return CCPAData.fromCCPAString(ccpaString)
    }

    /**
     * Set the CCPA privacy data.
     */
    fun setPrivacyData(apiVersion: Int, ccpaData: CCPAData) {
        checkApiVersion(apiVersion)
        val ccpaString = ccpaData.uspString
        if (CCPAStringValidator.isValidString(ccpaString)) {
            storage.putValue(privacyStringStorageKey, ccpaString)
        } else {
            throw CCPAException.invalidString(ccpaString)
        }
    }

    private fun checkApiVersion(apiVersion: Int) {
        if (apiVersion != supportedApiVersion) {
            throw CCPAException.invalidVersion(supportedApiVersion, apiVersion)
        }
    }

    private fun getStringFromStorage(): String {
        val storedString = storage.getValue(privacyStringStorageKey, "")
        if (storedString?.isNotBlank() == true) {
            val isValid = CCPAStringValidator.isValidString(storedString)
            if (isValid) {
                return storedString
            } else {
                debug("Stored CCPA String is invalid: $storedString")
                storage.deleteKey(privacyStringStorageKey)
            }
        }
        return CCPAStringValidator.initialValue
    }
}
