package com.usercentrics.sdk

import com.usercentrics.sdk.models.api.ApiConstants
import com.usercentrics.sdk.models.common.NetworkMode
import com.usercentrics.sdk.models.common.UsercentricsLoggerLevel
import kotlinx.serialization.Contextual
import kotlinx.serialization.Serializable

/** CMP configuration. Use the empty or [settingsId] constructor for convenience. */
@Serializable
class UsercentricsOptions(
    /** The [settingsId] is a Usercentrics generated ID, used to identify a unique CMP configuration. */
    var settingsId: String = "",

    /** The [defaultLanguage] sets the default language in the language selection hierarchy. E.g. "en", "de", "fr". The default value is empty. */
    var defaultLanguage: String = "",

    /** The [version] freezes the configuration version shown to your users, you may pass a specific version here. E.g. "3.0.4". The default value is "latest". */
    var version: String = ApiConstants.FALLBACK_VERSION,

    /** The [timeoutMillis] sets a timeout for network requests in milliseconds. The default value is 10000ms. */
    var timeoutMillis: Long = ApiConstants.DEFAULT_TIMEOUT_MILLIS,

    /** The [loggerLevel] provides a set of logs in the standard output depending on the Level: [.debug] (most detailed logs, includes every other level), [.warning] (non-problematic operations) and [.error] (relevant logs to any blocking problems). The default value is [.none]. */
    @Contextual var loggerLevel: UsercentricsLoggerLevel = UsercentricsLoggerLevel.NONE,

    /** The [ruleSetId] is a Usercentrics generated ID, used to identify a set of CMP configurations. */
    var ruleSetId: String = "",

    /** The [consentMediation] enables the feature of consent mediation, this feature works with a set of selected SDKs that you can find in the documentation. */
    var consentMediation: Boolean = false,

    /**The [domains] when successfully configured, enabled the self-hosting feature. You con find more about this on the documentation**/
    var domains: UsercentricsDomains? = null,

    /** The [initTimeoutMillis] sets a timeout for the SDK to initialize. Minimum value is 5000ms. The default value is 10000ms. */
    var initTimeoutMillis: Long = ApiConstants.DEFAULT_TIMEOUT_MILLIS,
) {

    /** The [networkMode] sets the network operation mode. Be careful, use this option only if we have confirmed that it is ready to use because it has a significant impact on the whole system's performance. The default value is "world". */
    @Contextual
    var networkMode: NetworkMode = NetworkMode.WORLD

    init {
        settingsId = settingsId.trim()
        defaultLanguage = defaultLanguage.trim()
        ruleSetId = ruleSetId.trim()
    }

    constructor() : this(defaultLanguage = "")

    constructor(settingsId: String) : this(settingsId = settingsId, defaultLanguage = "")

    internal fun copy(
        settingsId: String = this.settingsId,
        defaultLanguage: String = this.defaultLanguage,
        version: String = this.version,
        timeoutMillis: Long = this.timeoutMillis,
        loggerLevel: UsercentricsLoggerLevel = this.loggerLevel,
        ruleSetId: String = this.ruleSetId,
        networkMode: NetworkMode = this.networkMode,
        consentMediation: Boolean = this.consentMediation,
        domains: UsercentricsDomains? = this.domains,
        initTimeoutMillis: Long = this.initTimeoutMillis,
    ): UsercentricsOptions {
        return UsercentricsOptions(
            settingsId = settingsId,
            defaultLanguage = defaultLanguage,
            version = version,
            timeoutMillis = timeoutMillis,
            loggerLevel = loggerLevel,
            ruleSetId = ruleSetId,
            consentMediation = consentMediation,
            domains = domains?.copy(),
            initTimeoutMillis = initTimeoutMillis,
        ).also {
            it.networkMode = networkMode
        }
    }

    internal fun isSelfHostedConfigurationValid(): Boolean {
        return isSelfHostedConfigurationStatus(valid = true)
    }

    internal fun isSelfHostedConfigurationInvalid(): Boolean {
        return isSelfHostedConfigurationStatus(valid = false)
    }

    private fun isSelfHostedConfigurationStatus(valid: Boolean): Boolean {
        return this.domains != null && this.domains?.isValid() == valid
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as UsercentricsOptions

        if (settingsId != other.settingsId) return false
        if (defaultLanguage != other.defaultLanguage) return false
        if (version != other.version) return false
        if (timeoutMillis != other.timeoutMillis) return false
        if (loggerLevel != other.loggerLevel) return false
        if (ruleSetId != other.ruleSetId) return false
        if (consentMediation != other.consentMediation) return false
        if (domains != other.domains) return false
        if (initTimeoutMillis != other.initTimeoutMillis) return false
        return networkMode == other.networkMode
    }

    override fun hashCode(): Int {
        var result = settingsId.hashCode()
        result = 31 * result + defaultLanguage.hashCode()
        result = 31 * result + version.hashCode()
        result = 31 * result + timeoutMillis.hashCode()
        result = 31 * result + loggerLevel.hashCode()
        result = 31 * result + ruleSetId.hashCode()
        result = 31 * result + consentMediation.hashCode()
        result = 31 * result + (domains?.hashCode() ?: 0)
        result = 31 * result + initTimeoutMillis.hashCode()
        result = 31 * result + networkMode.hashCode()
        return result
    }
}
