package com.usercentrics.sdk

import com.usercentrics.ccpa.CCPAData
import com.usercentrics.sdk.errors.UsercentricsError
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.models.settings.PredefinedUIVariant
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.services.tcf.interfaces.TCFUserDecisions
import com.usercentrics.sdk.ui.PredefinedUIApplicationManager
import com.usercentrics.sdk.ui.PredefinedUIFactoryHolder

abstract class UsercentricsSDK {

    internal abstract suspend fun initialize(offlineMode: Boolean): Result<Unit>

    /** True, if consent has never been collected or some service has changed thus requiring a consent update. False, if consent has been collected and no update is required. */
    abstract fun shouldCollectConsent(): Boolean

    /** List of services with consent choices. */
    abstract fun getConsents(): List<UsercentricsServiceConsent>

    /** Get all the CMP Data. Use this data to build a custom UI CMP frontend. */
    abstract fun getCMPData(): UsercentricsCMPData

    /** Get a Usercentrics generated ID, used to identify a user's consent history. */
    abstract fun getControllerId(): String

    /** Get the User's Session Data that can be injected in a WebView with Usercentrics Browser SDK. */
    abstract fun getUserSessionData(): String

    /** Get the User's CCPA consent data. */
    abstract fun getUSPData(): CCPAData

    /** Set the CMP ID value required by IAB for custom UI. */
    abstract fun setCMPId(id: Int)

    /** Get the data that needs to be disclosed to the end-user if TCF is enabled. */
    abstract fun getTCFData(callback: (TCFData) -> Unit)

    /** Restore a user session from another Usercentrics supported platform. */
    abstract fun restoreUserSession(controllerId: String, onSuccess: (UsercentricsReadyStatus) -> Unit, onFailure: (UsercentricsError) -> Unit)

    /** Change the CMP language. */
    abstract fun changeLanguage(language: String, onSuccess: () -> Unit, onFailure: (UsercentricsError) -> Unit)

    /** Accept all services and TCF. */
    abstract fun acceptAllForTCF(fromLayer: TCFDecisionUILayer, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Accept all services. */
    abstract fun acceptAll(consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Deny all services and TCF. */
    abstract fun denyAllForTCF(fromLayer: TCFDecisionUILayer, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Deny all services. */
    abstract fun denyAll(consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Save service and TCF decisions. */
    abstract fun saveDecisionsForTCF(
        tcfDecisions: TCFUserDecisions,
        fromLayer: TCFDecisionUILayer,
        serviceDecisions: List<UserDecision>,
        consentType: UsercentricsConsentType
    ): List<UsercentricsServiceConsent>

    /** Save service decisions. */
    abstract fun saveDecisions(decisions: List<UserDecision>, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Save service and CCPA decisions. */
    abstract fun saveOptOutForCCPA(isOptedOut: Boolean, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent>

    /** Internal */
    abstract fun getUIApplication(predefinedUIVariant: PredefinedUIVariant): PredefinedUIApplicationManager

    /** Internal */
    abstract fun getUIFactoryHolder(
        abTestingVariant: String?,
        predefinedUIVariant: PredefinedUIVariant?,
        callback: (PredefinedUIFactoryHolder) -> Unit,
    )

    /** Track a CMP [event] from a custom UI CMP frontend. */
    abstract fun track(event: UsercentricsAnalyticsEventType)

    /** Set the A/B testing active variant of the CMP. */
    abstract fun setABTestingVariant(variantName: String)

    /** Get the A/B testing active variant of the CMP. */
    abstract fun getABTestingVariant(): String?

    /** Get Google's Additional Consent Mode Data */
    abstract fun getAdditionalConsentModeData(): AdditionalConsentModeData

    /** Clears current user session without needing to initialize the SDK again **/
    abstract fun clearUserSession(onSuccess: (UsercentricsReadyStatus) -> Unit, onError: (UsercentricsError) -> Unit)

    internal abstract fun readyStatus(): UsercentricsReadyStatus
}
