package com.usercentrics.sdk.acm.service

import com.usercentrics.sdk.AdTechProvider
import com.usercentrics.sdk.AdditionalConsentModeData
import com.usercentrics.sdk.acm.repository.AdditionalConsentModeRemoteRepository
import com.usercentrics.sdk.assert
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage

internal class AdditionalConsentModeServiceImpl(
    private val remoteRepository: AdditionalConsentModeRemoteRepository,
    private val deviceStorage: DeviceStorage,
    private val logger: UsercentricsLogger,
) : AdditionalConsentModeService {

    companion object {
        private const val LOAD_EMPTY_LIST = "Error: cannot load Google Additional Consent Mode without selecting any vendor on Admin Interface"
        private const val SAVE_ERROR = "Error when saving user consents for Google Additional Consent Mode. Cause: missing Ad Tech Provider list."
        private const val AC_STRING_VERSION = "2"
    }

    override var adTechProviderList: List<AdTechProvider>? = null
    override var acString: String? = null

    override suspend fun load(selectedIds: List<Int>) {
        if (selectedIds.isEmpty()) {
            logger.error(LOAD_EMPTY_LIST)
            assert(false) { LOAD_EMPTY_LIST }
            return
        }

        logger.debug("Loading Google Additional Consent Mode Providers $selectedIds")

        loadACString()
        val consentedIds = getConsentedIdsFromACString()

        remoteRepository.loadAdTechProviderList(selectedIds, consentedIds).also {
            this.adTechProviderList = it
        }
    }

    override fun save(acString: String) {
        if (acString.isBlank()) {
            return
        }

        storeACString(acString)

        if (this.adTechProviderList.isNullOrEmpty()) {
            return
        }

        val consentedIds = getConsentedIdsFromACString()
        updateAdTechProvidersWith(consentedIds)
    }

    override fun save(consentedIds: List<Int>) {
        if (canSaveConsents()) {
            updateAdTechProvidersWith(consentedIds)
            val acString = encodeACString()
            storeACString(acString)
        }
    }

    override fun acceptAll() {
        if (canSaveConsents()) {
            save(this.adTechProviderList!!.map { it.id })
        }
    }

    override fun denyAll() {
        if (canSaveConsents()) {
            save(emptyList())
        }
    }

    override fun getData(): AdditionalConsentModeData {
        val acString = this.acString
        if (acString.isNullOrBlank()) {
            return emptyData()
        }

        val adTechProviders = adTechProviderList
        if (adTechProviders.isNullOrEmpty()) {
            return emptyData()
        }
        return AdditionalConsentModeData(acString = acString, adTechProviders = adTechProviders)
    }

    override fun reset() {
        this.adTechProviderList = this.adTechProviderList?.map { it.copy(consent = false) }
        this.acString = encodeACString()
    }

    override fun didATPSChange(selectedIds: List<Int>): Boolean {
        return selectedIds != getStoredATPS()
    }

    private fun getStoredATPS(): List<Int> {
        val consentString = this.acString ?: ""

        return consentString
            .split("2~", "dv.", ".", "~")
            .mapNotNull { it.toIntOrNull() }
            .sorted()
    }

    private fun loadACString() {
        deviceStorage.getACString().also {
            this.acString = it
        }
    }

    private fun encodeACString(): String {
        val providers = this.adTechProviderList
        if (providers.isNullOrEmpty()) {
            return ""
        }

        val consentedString = StringBuilder()
        val disclosedString = StringBuilder()

        providers.forEach { provider ->
            val targetString = if (provider.consent) consentedString else disclosedString
            if (targetString.isNotEmpty()) {
                targetString.append(".")
            }
            targetString.append(provider.id)
        }

        if (consentedString.isNotEmpty()) {
            consentedString.append("~")
        }
        return "${AC_STRING_VERSION}~${consentedString}dv.${disclosedString}"
    }

    private fun getConsentedIdsFromACString(): List<Int> {
        val parts = acString?.split("~")
        if (parts?.size != 3) {
            return emptyList()
        }
        return parts[1].split(".").toList().mapNotNull { it.toIntOrNull() }
    }

    private fun storeACString(acString: String) {
        this.acString = acString
        deviceStorage.saveACString(acString)
    }

    private fun updateAdTechProvidersWith(consentedIds: List<Int>) {
        this.adTechProviderList = this.adTechProviderList?.map {
            it.copy(consent = consentedIds.contains(it.id))
        }
    }

    private fun canSaveConsents(): Boolean {
        if (this.adTechProviderList?.isNotEmpty() == true) {
            return true
        }

        logger.error(SAVE_ERROR)
        assert(false) { SAVE_ERROR }
        return false
    }

    private fun emptyData(): AdditionalConsentModeData {
        return AdditionalConsentModeData(acString = "", adTechProviders = emptyList())
    }
}
