package com.usercentrics.sdk.core.application

import com.usercentrics.sdk.*
import com.usercentrics.sdk.acm.api.AdditionalConsentModeApiImpl
import com.usercentrics.sdk.acm.repository.AdditionalConsentModeRemoteRepositoryImpl
import com.usercentrics.sdk.acm.service.AdditionalConsentModeService
import com.usercentrics.sdk.acm.service.AdditionalConsentModeServiceImpl
import com.usercentrics.sdk.core.ClassLocator
import com.usercentrics.sdk.core.NativeClassLocator
import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.v2.settings.service.CacheBypassResolver
import com.usercentrics.sdk.v2.settings.service.ICacheBypassResolver
import com.usercentrics.sdk.core.settings.SettingsOrchestrator
import com.usercentrics.sdk.core.settings.SettingsOrchestratorImpl
import com.usercentrics.sdk.domain.api.http.HttpClient
import com.usercentrics.sdk.domain.api.http.HttpRequests
import com.usercentrics.sdk.domain.api.http.HttpRequestsImpl
import com.usercentrics.sdk.extensions.emptyToNull
import com.usercentrics.sdk.lifecycle.ApplicationLifecycleListener
import com.usercentrics.sdk.lifecycle.BillingSessionLifecycleCallback
import com.usercentrics.sdk.lifecycle.LifecycleListenerProvider
import com.usercentrics.sdk.log.MainLoggerWriter
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.log.UsercentricsLoggerImpl
import com.usercentrics.sdk.mediation.facade.IMediationFacade
import com.usercentrics.sdk.mediation.facade.MediationFacade
import com.usercentrics.sdk.mediation.service.MediationServiceFactory
import com.usercentrics.sdk.models.api.ApiConstants
import com.usercentrics.sdk.models.common.NetworkMode
import com.usercentrics.sdk.models.common.UsercentricsLoggerLevel
import com.usercentrics.sdk.predefinedUI.PredefinedUIApplication
import com.usercentrics.sdk.services.api.BillingApi
import com.usercentrics.sdk.services.api.BillingApiImpl
import com.usercentrics.sdk.services.api.MainNetworkResolver
import com.usercentrics.sdk.services.api.NetworkResolver
import com.usercentrics.sdk.services.billing.BillingService
import com.usercentrics.sdk.services.billing.BillingServiceImpl
import com.usercentrics.sdk.services.ccpa.Ccpa
import com.usercentrics.sdk.services.ccpa.ICcpa
import com.usercentrics.sdk.services.dataFacade.DataFacade
import com.usercentrics.sdk.services.deviceStorage.*
import com.usercentrics.sdk.services.deviceStorage.migrations.*
import com.usercentrics.sdk.services.initialValues.InitialValuesStrategy
import com.usercentrics.sdk.services.initialValues.InitialValuesStrategyImpl
import com.usercentrics.sdk.services.initialValues.variants.CCPAStrategyImpl
import com.usercentrics.sdk.services.initialValues.variants.GDPRStrategyImpl
import com.usercentrics.sdk.services.initialValues.variants.TCFStrategyImpl
import com.usercentrics.sdk.services.settings.*
import com.usercentrics.sdk.services.tcf.TCF
import com.usercentrics.sdk.services.tcf.TCFUseCase
import com.usercentrics.sdk.ui.PredefinedUIMediator
import com.usercentrics.sdk.ui.PredefinedUIMediatorImpl
import com.usercentrics.sdk.ui.userAgent.NativeUserAgentProvider
import com.usercentrics.sdk.ui.userAgent.UserAgentProvider
import com.usercentrics.sdk.ui.userAgent.UserAgentSDKTypeEvaluatorImpl
import com.usercentrics.sdk.v2.analytics.api.AnalyticsApi
import com.usercentrics.sdk.v2.analytics.facade.AnalyticsFacade
import com.usercentrics.sdk.v2.analytics.facade.IAnalyticsFacade
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import com.usercentrics.sdk.v2.async.dispatcher.MainSemaphore
import com.usercentrics.sdk.v2.consent.api.GetConsentsApiImpl
import com.usercentrics.sdk.v2.consent.api.SaveConsentsApiImpl
import com.usercentrics.sdk.v2.consent.service.ConsentsService
import com.usercentrics.sdk.v2.consent.service.ConsentsServiceImpl
import com.usercentrics.sdk.v2.cookie.api.CookieInformationApi
import com.usercentrics.sdk.v2.cookie.repository.CookieInformationRepository
import com.usercentrics.sdk.v2.cookie.service.CookieInformationService
import com.usercentrics.sdk.v2.cookie.service.UsercentricsCookieInformationService
import com.usercentrics.sdk.v2.etag.cache.EtagCacheStorage
import com.usercentrics.sdk.v2.etag.cache.IEtagCacheStorage
import com.usercentrics.sdk.v2.file.FileStorageResolver
import com.usercentrics.sdk.v2.file.IFileStorage
import com.usercentrics.sdk.v2.language.api.LanguageApi
import com.usercentrics.sdk.v2.language.facade.ILanguageFacade
import com.usercentrics.sdk.v2.language.facade.LanguageFacade
import com.usercentrics.sdk.v2.language.repository.LanguageRepository
import com.usercentrics.sdk.v2.language.service.ILanguageService
import com.usercentrics.sdk.v2.language.service.LanguageService
import com.usercentrics.sdk.v2.location.cache.LocationCache
import com.usercentrics.sdk.v2.location.repository.LocationRepository
import com.usercentrics.sdk.v2.location.service.ILocationService
import com.usercentrics.sdk.v2.location.service.LocationService
import com.usercentrics.sdk.v2.ruleset.api.RuleSetApi
import com.usercentrics.sdk.v2.ruleset.repository.RuleSetRepository
import com.usercentrics.sdk.v2.ruleset.service.IRuleSetService
import com.usercentrics.sdk.v2.ruleset.service.RuleSetService
import com.usercentrics.sdk.v2.settings.api.AggregatorApi
import com.usercentrics.sdk.v2.settings.api.SettingsApi
import com.usercentrics.sdk.v2.settings.facade.SettingsFacade
import com.usercentrics.sdk.v2.settings.repository.AggregatorRepository
import com.usercentrics.sdk.v2.settings.repository.SettingsRepository
import com.usercentrics.sdk.v2.settings.service.ISettingsService
import com.usercentrics.sdk.v2.settings.service.SettingsService
import com.usercentrics.sdk.v2.tcf.api.TCFDeclarationsApi
import com.usercentrics.sdk.v2.tcf.api.TCFVendorListApi
import com.usercentrics.sdk.v2.tcf.facade.TCFFacadeImpl
import com.usercentrics.sdk.v2.tcf.repository.TCFDeclarationsRepository
import com.usercentrics.sdk.v2.tcf.repository.TCFVendorListRepository
import com.usercentrics.sdk.v2.tcf.service.ITCFService
import com.usercentrics.sdk.v2.tcf.service.TCFService
import com.usercentrics.sdk.v2.translation.api.TranslationApi
import com.usercentrics.sdk.v2.translation.repository.TranslationRepository
import com.usercentrics.sdk.v2.translation.service.ITranslationService
import com.usercentrics.sdk.v2.translation.service.TranslationService
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers

internal open class MainApplication(
    private val options: UsercentricsOptions,
    appContext: UsercentricsContext?
) : Application {

    private val cacheId = options.ruleSetId.emptyToNull() ?: options.settingsId
    private val networkMode: NetworkMode = options.networkMode

    override var networkStrategy: Lazy<INetworkStrategy> = lazy { NetworkStrategyImpl() }

    override val userAgentProvider: UserAgentProvider by lazy {
        val userAgentSDKTypeEvaluator = UserAgentSDKTypeEvaluatorImpl(classLocator = classLocator.value, sdkVersion = BuildKonfig.sdk_version)
        NativeUserAgentProvider(
            context = appContext,
            userAgentSDKTypeEvaluator = userAgentSDKTypeEvaluator,
            predefinedUIMediator = predefinedUIMediator,
            options = options,
        )
    }

    private val timeoutMillis: Long by lazy { options.timeoutMillis }

    private val storageProvider: KeyValueStorageProvider by lazy {
        KeyValueStorageProvider(appContext)
    }

    override var httpClient: Lazy<HttpClient> = lazy {
        UsercentricsApplication.provideHttpClient(timeoutMillis, dispatcher)
    }

    override var networkResolver: Lazy<NetworkResolver> = lazy {
        val domains = options.domains ?: EmptyUsercentricsDomains()
        MainNetworkResolver(networkMode = networkMode, domains = domains)
    }

    override val httpInstance: HttpRequests by lazy {
        HttpRequestsImpl(httpClient.value, userAgentProvider, dispatcher)
    }

    override val logger: UsercentricsLogger by lazy {
        buildLogger(options.loggerLevel)
    }

    override val billingApi: BillingApi by lazy {
        BillingApiImpl(httpInstance, networkResolver.value, userAgentProvider.provide().appID)
    }

    override val languageService: ILanguageService by lazy {
        val languageApi = LanguageApi(httpInstance, networkResolver.value)
        val languageRepository = LanguageRepository(languageApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        LanguageService(languageRepository, storageInstance.value, logger)
    }

    override val settingsService: ISettingsService by lazy {
        val settingsApi = SettingsApi(httpInstance, networkResolver.value)
        val settingsRepository = SettingsRepository(settingsApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        val aggregatorApi = AggregatorApi(logger, networkResolver.value, httpInstance)
        val aggregatorRepository = AggregatorRepository(aggregatorApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        SettingsService(settingsRepository, aggregatorRepository, cacheBypassProvider)
    }

    override val cookieInformationService: UsercentricsCookieInformationService by lazy {
        val cookieInformationApi = CookieInformationApi(httpInstance)
        val repository = CookieInformationRepository(cookieInformationApi, jsonParserInstance)
        CookieInformationService(dispatcher, tcfService, repository, settingsInstance.value)
    }

    override val translationService: ITranslationService by lazy {
        val translationApi = TranslationApi(httpInstance, networkResolver.value)
        val translationRepository = TranslationRepository(translationApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        TranslationService(translationRepository)
    }

    override val settingsFacade: SettingsFacade by lazy {
        val serviceSettingsMapper = SettingsServicesMapper(generatorIds.value)
        val settingsMapper = SettingsMapper(logger, serviceSettingsMapper, generatorIds.value)
        SettingsFacade(settingsService, translationService, settingsMapper, cacheBypassProvider)
    }

    private val cacheBypassProvider: ICacheBypassResolver by lazy {
        CacheBypassResolver()
    }

    override var consentsService: Lazy<ConsentsService> = lazy {
        val getConsentsApi = GetConsentsApiImpl(httpInstance, networkResolver.value, jsonParserInstance, settingsOrchestrator.value)
        val saveConsentsApi = SaveConsentsApiImpl(httpInstance, networkResolver.value, jsonParserInstance, userAgentProvider)
        ConsentsServiceImpl(
            dispatcher = dispatcher,
            logger = logger,
            getConsentsApi = getConsentsApi,
            saveConsentsApi = saveConsentsApi,
            deviceStorage = storageInstance.value,
            settingsService = settingsService,
            settingsLegacyInstance = settingsInstance.value,
        )
    }

    override var initialValuesStrategy: Lazy<InitialValuesStrategy> = lazy {
        val ccpaStrategy = CCPAStrategyImpl(logger, storageInstance.value, ccpa = ccpaInstance.value)
        val tcfStrategy = TCFStrategyImpl(logger, storageInstance.value)
        val gdprStrategy = GDPRStrategyImpl(logger, storageInstance.value)

        InitialValuesStrategyImpl(
            dataFacade = dataFacadeInstance,
            deviceStorage = storageInstance.value,
            settingsLegacy = settingsInstance.value,
            locationService = locationService.value,
            tcf = tcfInstance.value,
            ccpaStrategy = ccpaStrategy,
            tcfStrategy = tcfStrategy,
            gdprStrategy = gdprStrategy,
            settingsOrchestrator = settingsOrchestrator.value,
            additionalConsentModeService = additionalConsentModeService.value,
            logger = logger
        )
    }

    override val uiDependencyManager: PredefinedUIApplication by lazy {
        PredefinedUIApplication(cookieInformationService, logger, options.loggerLevel)
    }

    override var lifecycleListener: Lazy<ApplicationLifecycleListener> = lazy {
        LifecycleListenerProvider().provide(billingSessionLifecycleCallback)
    }

    override val billingSessionLifecycleCallback: BillingSessionLifecycleCallback by lazy {
        BillingSessionLifecycleCallback(billingService.value, settingsOrchestrator.value)
    }

    override var defaultKeyValueStorage: Lazy<KeyValueStorage> = lazy {
        storageProvider.provideDefault()
    }

    override var customKeyValueStorage: Lazy<KeyValueStorage> = lazy {
        storageProvider.provideCustom(ApiConstants.USERCENTRICS_PREFERENCES_NAME)
    }

    override var storageInstance: Lazy<DeviceStorage> = lazy {
        val storageHolder = StorageHolder(defaultKeyValueStorage.value, customKeyValueStorage.value)
        val builder = UsercentricsDeviceStorage.Builder(
            storageHolder = storageHolder,
            logger = logger,
            jsonParser = jsonParserInstance
        )
        builder
            .addMigration(MigrationToVersion1(storageHolder, jsonParserInstance))
            .addMigration(MigrationToVersion2(storageHolder))
            .addMigration(MigrationToVersion3(storageHolder, jsonParserInstance, isTVOS))
            .addMigration(MigrationToVersion4(storageHolder))
            .addMigration(MigrationToVersion5(storageHolder))
            .addMigration(MigrationToVersion6(storageHolder, jsonParserInstance))
            .addMigration(MigrationToVersion7(storageHolder))
            .addMigration(MigrationToVersion8(storageHolder, jsonParserInstance, fileStorage.value))
            .build()
    }

    override var billingService: Lazy<BillingService> = lazy {
        BillingServiceImpl(dispatcher, storageInstance.value, billingApi)
    }

    override var languageFacade: Lazy<ILanguageFacade> = lazy {
        LanguageFacade(languageService)
    }

    override var locationService: Lazy<ILocationService> = lazy {
        LocationService(locationRepository)
    }

    private val locationCache by lazy { LocationCache(customKeyValueStorage.value) }
    private val locationRepository by lazy { LocationRepository(locationCache, jsonParserInstance) }

    override var settingsInstance: Lazy<ISettingsLegacy> = lazy {
        SettingsLegacy(settingsFacade, generatorIds.value)
    }

    override var generatorIds: Lazy<IGeneratorIds> = lazy {
        GeneratorIds()
    }

    override val dataFacadeInstance: DataFacade by lazy {
        DataFacade(
            consentsService = consentsService.value,
            settingsInstance = settingsInstance.value,
            settingsService = settingsService,
            storageInstance = storageInstance.value,
            tcfInstance = tcfInstance.value,
            additionalConsentModeService = additionalConsentModeService.value,
            logger = logger,
        )
    }

    override var ccpaInstance: Lazy<ICcpa> = lazy {
        Ccpa(storageInstance.value, logger)
    }

    private val tcfService: ITCFService by lazy {
        val tcfDeclarationsApi = TCFDeclarationsApi(httpInstance, networkResolver.value)
        val tcfVendorListApi = TCFVendorListApi(httpInstance, networkResolver.value)

        TCFService(
            TCFVendorListRepository(tcfVendorListApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value),
            TCFDeclarationsRepository(tcfDeclarationsApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        )
    }

    override var tcfInstance: Lazy<TCFUseCase> = lazy {
        val tcfFacade = TCFFacadeImpl(tcfService)
        val semaphore = MainSemaphore()

        TCF(
            logger = logger,
            settingsService = settingsService,
            storageInstance = storageInstance.value,
            consentsService = consentsService.value,
            locationService = locationService.value,
            additionalConsentModeService = additionalConsentModeService.value,
            tcfFacade = tcfFacade,
            dispatcher = dispatcher,
            semaphore = semaphore,
            settingsOrchestrator = settingsOrchestrator.value,
        )
    }

    override val jsonParserInstance: JsonParser by lazy {
        JsonParser()
    }

    override val mainDispatcher: CoroutineDispatcher by lazy {
        Dispatchers.Main
    }

    override val defaultDispatcher: CoroutineDispatcher by lazy {
        Dispatchers.Default
    }

    override val dispatcher: Dispatcher by lazy {
        Dispatcher(mainDispatcher, defaultDispatcher)
    }

    override var fileStorage: Lazy<IFileStorage> = lazy {
        FileStorageResolver().buildFileStorage(appContext)
    }

    override val analyticsFacade: Lazy<IAnalyticsFacade> = lazy {
        val api = AnalyticsApi(networkResolver.value, httpInstance, userAgentProvider.provide().appID)
        AnalyticsFacade(api, settingsService, dispatcher, logger)
    }

    override var classLocator: Lazy<ClassLocator> = lazy {
        NativeClassLocator()
    }

    override val predefinedUIMediator: PredefinedUIMediator by lazy {
        PredefinedUIMediatorImpl(classLocator.value, customKeyValueStorage.value)
    }

    override var etagCacheStorage: Lazy<IEtagCacheStorage> = lazy {
        EtagCacheStorage(fileStorage.value, dispatcher)
    }

    override var settingsOrchestrator: Lazy<SettingsOrchestrator> = lazy {
        SettingsOrchestratorImpl(this)
    }

    override var ruleSetService: Lazy<IRuleSetService> = lazy {
        val ruleSetApi = RuleSetApi(logger, networkResolver.value, httpInstance)
        val ruleSetRepository = RuleSetRepository(ruleSetApi, jsonParserInstance, logger, etagCacheStorage.value, networkStrategy.value)
        RuleSetService(ruleSetRepository, locationService.value)
    }

    override var mediationFacade: Lazy<IMediationFacade> = lazy {
        val mediationServiceFactory = MediationServiceFactory(logger = logger, context = appContext)
        MediationFacade(mediationService = mediationServiceFactory.build(), logger = logger)
    }

    override var additionalConsentModeService: Lazy<AdditionalConsentModeService> = lazy {
        val api = AdditionalConsentModeApiImpl(
            restClient = httpInstance,
            networkResolver = networkResolver.value
        )
        val repository = AdditionalConsentModeRemoteRepositoryImpl(
            api = api,
            jsonParser = jsonParserInstance,
            logger = logger,
            etagCacheStorage = etagCacheStorage.value,
            networkStrategy = networkStrategy.value
        )

        AdditionalConsentModeServiceImpl(
            remoteRepository = repository,
            deviceStorage = storageInstance.value,
            logger = logger,
        )
    }

    override fun boot() {
        etagCacheStorage.value.boot(cacheId)

        if (!options.isSelfHostedConfigurationValid()) {
            lifecycleListener.value.setup()
            billingService.value.dispatchSessionBuffer()
        }

        consentsService.value.processConsentsBuffer()
    }

    override fun tearDown(clearStorage: Boolean) {
        lifecycleListener.value.tearDown()
        if (clearStorage) {
            dispatcher.dispatch { fileStorage.value.rmAll() }
            storageInstance.value.clear()
        }
    }

    private fun buildLogger(loggerLevel: UsercentricsLoggerLevel): UsercentricsLogger {
        return UsercentricsLoggerImpl(
            level = loggerLevel,
            MainLoggerWriter()
        )
    }
}
