package com.usercentrics.sdk.core.json

import com.usercentrics.sdk.log.UsercentricsLogger
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.SerializationStrategy
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.decodeFromJsonElement
import kotlin.native.concurrent.ThreadLocal

//@ThreadLocal
private val json = Json {
    encodeDefaults = true
    ignoreUnknownKeys = true
    coerceInputValues = true
    allowSpecialFloatingPointValues = false
}

internal class JsonParser {

    inline fun <reified T> decodeFromString(str: String): T {
        return json.decodeFromString(str)
    }

    inline fun <reified T> decodeFromString(deserializer: DeserializationStrategy<T>, str: String): T {
        return json.decodeFromString(deserializer, str)
    }

    inline fun <reified T> tryToDecodeFromString(deserializer: DeserializationStrategy<T>, str: String, logger: UsercentricsLogger? = null): T? {
        return json.tryToDecodeFromString(deserializer, str, logger)
    }

    inline fun <reified T> encodeToString(input: T): String {
        return json.encodeToString(input)
    }

    inline fun <reified T> encodeToString(serializer: SerializationStrategy<T>, input: T): String {
        return json.encodeToString(serializer, input)
    }

    inline fun <reified T> decodeFromJsonElement(jsonElement: JsonElement): T {
        return json.decodeFromJsonElement(jsonElement)
    }
}

internal fun <T> Json.tryToDecodeFromString(
    deserializer: DeserializationStrategy<T>,
    string: String,
    logger: UsercentricsLogger? = null
): T? {
    return try {
        decodeFromString(deserializer, string)
    } catch (e: Throwable) {
        logger?.error(e.message ?: "Json parse error", e)
        null
    }
}
