package com.usercentrics.sdk.models.settings

import com.usercentrics.sdk.services.tcf.interfaces.IdAndName
import com.usercentrics.tcf.core.model.gvl.RetentionPeriod

internal object PredefinedUIDataTV {
    const val CONTENT_SEPARATOR = "\n\n"
    const val BULLET_POINT = '\u2022'
}

class PredefinedTVViewSettings(
    val customization: PredefinedUICustomization,
    val firstLayer: PredefinedTVFirstLayerSettings,
    val secondLayer: PredefinedTVSecondLayerSettings,
    val ariaLabels: PredefinedUIAriaLabels,
)

class PredefinedTVFirstLayerSettings(
    val title: String,
    val content: String,
    val logoUrl: String?,
    val primaryActions: List<PredefinedTVActionButton>,
    val secondaryActions: List<PredefinedTVActionButton>,
)

data class PredefinedTVActionButton(
    val label: String,
    val type: PredefinedTVActionButtonType,
)

sealed class PredefinedTVActionButtonType {
    object AcceptAll : PredefinedTVActionButtonType()
    object DenyAll : PredefinedTVActionButtonType()
    data class More(val initialSection: String? = null) : PredefinedTVActionButtonType()
    data class Url(val url: String) : PredefinedTVActionButtonType()
}

class PredefinedTVSecondLayerSettings(
    val title: String, // Privacy Settings
    val save: String, // Save & Close
    val denyAll: String?, // Deny All
    val toggleLocalizations: PredefinedUIToggleLocalizations,
    val content: PredefinedTVSecondLayerSettingsContent, // All Categories Content
)

class PredefinedUIToggleLocalizations(
    private val onText: String,
    private val offText: String
) {
    fun getTextForState(state: Boolean): String {
        return if (state) onText else offText
    }
}

sealed class PredefinedTVSecondLayerSettingsContent {

    data class Cards(
        val sections: List<PredefinedTVSecondLayerCardSection>
    ) : PredefinedTVSecondLayerSettingsContent()

    interface Details {
        val extraToggle: PredefinedUIToggleSettings?
        val entries: List<PredefinedTVSecondLayerDetailsEntry>
    }

    data class DetailsWithVendors(
        val title: String,
        override val extraToggle: PredefinedUIToggleSettings?,
        override val entries: List<PredefinedTVSecondLayerDetailsEntry>,
    ) : PredefinedTVSecondLayerSettingsContent(), Details

    data class DetailsWithIllustrations(
        override val extraToggle: PredefinedUIToggleSettings?,
        override val entries: List<PredefinedTVSecondLayerDetailsEntry>
    ) : PredefinedTVSecondLayerSettingsContent(), Details
}

data class PredefinedTVSecondLayerCardSection(
    val name: String,
    val entries: List<PredefinedTVSecondLayerCardEntry>,
)

data class PredefinedTVSecondLayerCardEntry(
    val id: String,
    val title: String, // Marketing/Firebase
    val description: String?,
    val toggle: PredefinedUIToggleSettings?,
    val content: PredefinedTVSecondLayerSettingsContent // All Services/Service Details
)

sealed class PredefinedTVSecondLayerDetailsEntry {
    companion object {

        internal fun mapPurposesWithRetention(idAndNameList: List<IdAndName>, retentionPeriod: RetentionPeriod? = null, label: String): String {
            return idAndNameList.joinToString(postfix = ".") { idAndName ->
                val time = retentionPeriod?.idAndPeriod?.get(idAndName.id)
                if (time != null) {
                    "${idAndName.name.trim()} ($label: ${time})"
                } else {
                    idAndName.name.trim()
                }
            }
        }

        // The join list cannot be a bullet list because we could run out of vertical space
        // and the vertical scroll works entry by entry (not line by line).
        internal fun mapContent(list: Iterable<String>, introduction: String? = null): String {
            return if (introduction == null) {
                list.joinToString(postfix = ".")
            } else {
                "$introduction\n\n${list.joinToString(postfix = ".")}"
            }
        }

        internal fun mapContentIllustrations(list: Iterable<String>): String {
            return list.joinToString(separator = PredefinedUIDataTV.CONTENT_SEPARATOR) { illustration -> "${PredefinedUIDataTV.BULLET_POINT} $illustration" }
        }
    }

    data class TitleContent(
        val title: String,
        val content: String,
    ) : PredefinedTVSecondLayerDetailsEntry()

    data class Text(
        val text: String,
    ) : PredefinedTVSecondLayerDetailsEntry()

    data class Link(
        val title: String,
        val linkLabel: String,
        val url: String,
    ) : PredefinedTVSecondLayerDetailsEntry()
}
