package com.usercentrics.sdk.models.settings

import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.extensions.arrayOfNotEmpty
import com.usercentrics.sdk.ui.color.UsercentricsShadedColor
import kotlin.math.floor

class PredefinedUICustomization(
    val color: PredefinedUICustomizationColor,
    val font: PredefinedUICustomizationFont,
    val logoUrl: String?,
    val cornerRadius: Int
)

class PredefinedUICustomizationColor(
    val text: UsercentricsShadedColor,
    val acceptAllButton: PredefinedUICustomizationColorButton,
    val denyAllButton: PredefinedUICustomizationColorButton,
    val saveButton: PredefinedUICustomizationColorButton,
    val okButton: PredefinedUICustomizationColorButton,
    val manageButton: PredefinedUICustomizationColorButton,
    val toggles: PredefinedUICustomizationColorToggles,
    val layerBackgroundColor: String,
    val layerBackgroundSecondaryColor: String,
    val linkColor: String,
    val tabColor: String,
    val baseOverlayColor: String,
    val overlayOpacity: Double,
    val tabsBorderColor: String,
) {
    internal companion object {
        const val defaultText = "#303030"
        const val defaultLegacyPrimary = "#0045A5"
        const val defaultAcceptAllButtonText = "#FAFAFA"
        const val defaultAcceptAllButtonBackground = "#0045A5"
        const val defaultDenyAllButtonText = "#FAFAFA"
        const val defaultDenyAllButtonBackground = "#0045A5"
        const val defaultSaveButtonText = "#303030"
        const val defaultSaveButtonBackground = "#F5F5F5"
        const val defaultOkButtonText = "#303030"
        const val defaultOkButtonBackground = "#F5F5F5"
        const val defaultButtonCornerRadius = 4
        const val defaultToggleIconColor = "#FFFFFF"
        const val defaultBackgroundColor = "#FFFFFF"
        const val defaultLinkColor = "#303030"
        const val defaultOverlayColor = "#333333"
        const val defaultTabsBorderColor = "#DDDDDD"
    }
}

class PredefinedUICustomizationColorToggles(
    val activeBackground: String,
    val inactiveBackground: String,
    val disabledBackground: String,

    val activeIcon: String,
    val inactiveIcon: String,
    val disabledIcon: String
)

class PredefinedUICustomizationColorButton(
    val text: String,
    val background: String,
    val cornerRadius: Int
)

class PredefinedUICustomizationFont(
    val family: String = defaultFamily,
    val size: Int = defaultSize
) {
    companion object {
        const val defaultFamily = ""
        const val defaultSize = 14
    }
}

data class PredefinedUILink(
    val label: String,
    val url: String?,
    val linkType: PredefinedUILinkType,
    val eventType: UsercentricsAnalyticsEventType,
) {
    companion object {
        fun legalLinkUrl(label: String, url: String?, eventType: UsercentricsAnalyticsEventType): PredefinedUILink {
            return PredefinedUILink(
                label = label,
                url = url,
                linkType = PredefinedUILinkType.URL,
                eventType = eventType,
            )
        }

        fun moreLink(label: String): PredefinedUILink {
            return PredefinedUILink(
                label = label,
                url = null,
                linkType = PredefinedUILinkType.MANAGE_SETTINGS,
                eventType = UsercentricsAnalyticsEventType.MORE_INFORMATION_LINK,
            )
        }
    }

    internal fun isEmpty(): Boolean {
        return label.isBlank() || (linkType == PredefinedUILinkType.URL && url.isNullOrBlank())
    }
}

internal class LegacyPoweredBy(
    val isEnabled: Boolean,
    val label: String = "Powered by", // TODO label needed
    val urlLabel: String = "Usercentrics Consent Management" // TODO label needed
)

class PredefinedUIGeneralLabels(
    val controllerId: String,
    val date: String,
    val decision: String,
    val readMore: String,
    val more: String,
    val acceptAll: String,
    val denyAll: String,
    val continueWithoutAccepting: String
)

class PredefinedUIServiceLabels(
    val dataCollected: PredefinedUIDescriptionTitle,
    val dataDistribution: PredefinedUIDataDistributionTitle,
    val dataPurposes: PredefinedUIDescriptionTitle,
    val dataRecipientsTitle: String,
    val descriptionTitle: String,
    val history: PredefinedUIDescriptionTitle,
    val legalBasis: PredefinedUIDescriptionTitle,
    val processingCompanyTitle: String,
    val retentionPeriodTitle: String,
    val technologiesUsed: PredefinedUIDescriptionTitle,
    val urls: PredefinedUIURLsTitle
)

// TODO Rename this since it is not exclusive for first layer anymore
class FirstLayerButtonLabels(
    val acceptAll: String,
    val denyAll: String,
    val more: String,
    val save: String,
)

class PredefinedUIDescriptionTitle(
    val titleDescription: String,
    val title: String
)

class PredefinedUIDataDistributionTitle(
    val processingLocationTitle: String,
    val thirdPartyCountriesTitle: String,
    val thirdPartyCountriesDescription: String,
)


// This legacy class is incomplete and problematic because it mix different type of content
// We should delete it and use UsercentricsLabels directly.
class PredefinedUIURLsTitle(
    val cookiePolicyTitle: String, // Click here...
    val dataProcessingAgreementTitle: String, // This one is a title and not a 'Click here...' long text
    val optOutTitle: String, // Click here...
    val privacyPolicyTitle: String // Click here...
)

data class PredefinedUICookieInformationLabels(
    val anyDomain: String,
    val day: String,
    val days: String,
    val domain: String,
    val duration: String,
    val error: String,
    val hour: String,
    val hours: String,
    val identifier: String,
    val loading: String,
    val maximumAge: String,
    val minute: String,
    val minutes: String,
    val month: String,
    val months: String,
    val multipleDomains: String,
    val no: String,
    val nonCookieStorage: String,
    val second: String,
    val seconds: String,
    val session: String,
    val title: String,
    val titleDetailed: String,
    val tryAgain: String,
    val type: String,
    val year: String,
    val years: String,
    val yes: String,
    val storageInformationDescription: String,
    val cookieStorage: String,
    val cookieRefresh: String,
    val purposes: String
) {

    companion object {
        private const val ONE_MINUTE_IN_SECONDS = 60.0
        private const val ONE_HOUR_IN_SECONDS = 60.0 * 60.0
        private const val ONE_DAY_IN_SECONDS = ONE_HOUR_IN_SECONDS * 24
        private const val ONE_MONTH_IN_SECONDS = 2628000.0
        private const val ONE_YEAR_IN_SECONDS = ONE_MONTH_IN_SECONDS * 12
    }

    fun cookieMaxAgeLabel(timestampInSeconds: Double): String {
        if (timestampInSeconds <= 0) {
            return session
        }
        val hasMoreThanOneDay = timestampInSeconds >= ONE_DAY_IN_SECONDS
        if (hasMoreThanOneDay) {
            return formatTimestampInYearsMonthsAndDays(timestampInSeconds)
        }
        return formatTimestampInHoursMinutesAndSeconds(timestampInSeconds)
    }

    private fun formatTimestampInHoursMinutesAndSeconds(timestampInSeconds: Double): String {
        var timestamp = timestampInSeconds

        val hoursLabel = getLabelOfTimestampValue(timestamp, ONE_HOUR_IN_SECONDS, this.hours, this.hour)
        timestamp %= ONE_HOUR_IN_SECONDS

        val minutesLabel = getLabelOfTimestampValue(timestamp, ONE_MINUTE_IN_SECONDS, this.minutes, this.minute)
        timestamp %= ONE_MINUTE_IN_SECONDS

        val secondsLabel = if (timestamp > 0.0) labelOfAmount(timestamp.toInt(), this.seconds, this.second) else ""
        return arrayOfNotEmpty(hoursLabel, minutesLabel, secondsLabel).joinToString()
    }

    private fun formatTimestampInYearsMonthsAndDays(timestampInSeconds: Double): String {
        var timestamp = timestampInSeconds

        val yearsLabel = getLabelOfTimestampValue(timestamp, ONE_YEAR_IN_SECONDS, this.years, this.year)
        timestamp %= ONE_YEAR_IN_SECONDS

        val monthsLabel = getLabelOfTimestampValue(timestamp, ONE_MONTH_IN_SECONDS, this.months, this.month)
        timestamp %= ONE_MONTH_IN_SECONDS

        val daysLabel = getLabelOfTimestampValue(timestamp, ONE_DAY_IN_SECONDS, this.days, this.day)
        return arrayOfNotEmpty(yearsLabel, monthsLabel, daysLabel).joinToString()
    }

    private fun getLabelOfTimestampValue(timestampInSeconds: Double, threshold: Double, pluralLabel: String, singularLabel: String): String {
        val amount = floor(timestampInSeconds / threshold).toInt()
        if (amount <= 0) {
            return ""
        }
        return labelOfAmount(amount, pluralLabel, singularLabel)
    }

    private fun labelOfAmount(amount: Int, pluralLabel: String, singularLabel: String): String {
        val label = if (amount > 1) pluralLabel else singularLabel
        return "$amount $label"
    }
}

class PredefinedUIAriaLabels(
    val acceptAllButton: String? = null,
    val ccpaButton: String? = null,
    val ccpaMoreInformation: String? = null,
    val closeButton: String? = null,
    val collapse: String? = null,
    val cookiePolicyButton: String? = null,
    val copyControllerId: String? = null,
    val denyAllButton: String? = null,
    val expand: String? = null,
    val fullscreenButton: String? = null,
    val imprintButton: String? = null,
    val languageSelector: String? = null,
    val privacyButton: String? = null,
    val privacyPolicyButton: String? = null,
    val saveButton: String? = null,
    val serviceInCategoryDetails: String? = null,
    val servicesInCategory: String? = null,
    val tabButton: String? = null,
    val usercentricsCMPButtons: String? = null,
    val usercentricsCMPContent: String? = null,
    val usercentricsCMPHeader: String? = null,
    val usercentricsCMPUI: String? = null,
    val usercentricsCard: String? = null,
    val usercentricsList: String? = null,
    val vendorConsentToggle: String? = null,
    val vendorDetailedStorageInformation: String? = null,
    val vendorLegIntToggle: String? = null,
    val logoAltTag: String? = null,
)
