package com.usercentrics.sdk.predefinedUI

import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.UsercentricsSDK
import com.usercentrics.sdk.UsercentricsServiceConsent
import com.usercentrics.sdk.assert
import com.usercentrics.sdk.models.common.UsercentricsVariant
import com.usercentrics.sdk.models.settings.PredefinedUIDecision
import com.usercentrics.sdk.models.settings.ServicesIdStrategy
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.ui.PredefinedUIInteraction
import com.usercentrics.sdk.ui.PredefinedUIResponse

internal class PredefinedUIConsentManagerImpl(
    private val usercentricsSDK: UsercentricsSDK,
    private val variant: UsercentricsVariant,
    private val controllerId: String
) : PredefinedUIConsentManager {

    override fun acceptAll(fromLayer: TCFDecisionUILayer): PredefinedUIResponse {
        val consents = when (variant) {
            UsercentricsVariant.DEFAULT -> acceptAllDefault()
            UsercentricsVariant.CCPA -> acceptAllCCPA()
            UsercentricsVariant.TCF -> acceptAllTCF(fromLayer)
        }

        val eventType = if (fromLayer == TCFDecisionUILayer.FIRST_LAYER) {
            UsercentricsAnalyticsEventType.ACCEPT_ALL_FIRST_LAYER
        } else {
            UsercentricsAnalyticsEventType.ACCEPT_ALL_SECOND_LAYER
        }
        trackAnalyticsEvent(eventType)

        return PredefinedUIResponse(
            userInteraction = PredefinedUIInteraction.ACCEPT_ALL,
            consents = consents,
            controllerId = controllerId,
        )
    }

    private fun acceptAllDefault(): List<UsercentricsServiceConsent> {
        return usercentricsSDK.acceptAll(consentType = UsercentricsConsentType.EXPLICIT)
    }

    private fun acceptAllCCPA(): List<UsercentricsServiceConsent> {
        return usercentricsSDK.saveOptOutForCCPA(
            isOptedOut = false,
            consentType = UsercentricsConsentType.EXPLICIT
        )
    }

    private fun acceptAllTCF(fromLayer: TCFDecisionUILayer?): List<UsercentricsServiceConsent> {
        assert(fromLayer != null) { "Failed while trying to accept all services: fromLayer parameter cannot be null" }

        return usercentricsSDK.acceptAllForTCF(
            fromLayer = fromLayer ?: TCFDecisionUILayer.FIRST_LAYER,
            consentType = UsercentricsConsentType.EXPLICIT
        )
    }

    override fun denyAll(fromLayer: TCFDecisionUILayer): PredefinedUIResponse {
        val consents = when (variant) {
            UsercentricsVariant.DEFAULT -> denyAllDefault()
            UsercentricsVariant.CCPA -> denyAllCCPA()
            UsercentricsVariant.TCF -> denyAllTCF(fromLayer)
        }

        val eventType = if (fromLayer == TCFDecisionUILayer.FIRST_LAYER) {
            UsercentricsAnalyticsEventType.DENY_ALL_FIRST_LAYER
        } else {
            UsercentricsAnalyticsEventType.DENY_ALL_SECOND_LAYER
        }
        trackAnalyticsEvent(eventType)

        return PredefinedUIResponse(
            userInteraction = PredefinedUIInteraction.DENY_ALL,
            consents = consents,
            controllerId = controllerId,
        )
    }

    private fun denyAllDefault(): List<UsercentricsServiceConsent> {
        return usercentricsSDK.denyAll(consentType = UsercentricsConsentType.EXPLICIT)
    }

    private fun denyAllCCPA(): List<UsercentricsServiceConsent> {
        return usercentricsSDK.saveOptOutForCCPA(
            isOptedOut = true,
            consentType = UsercentricsConsentType.EXPLICIT
        )
    }

    private fun denyAllTCF(fromLayer: TCFDecisionUILayer?): List<UsercentricsServiceConsent> {
        assert(fromLayer != null) { "Failed while trying to deny all services: fromLayer parameter cannot be null" }

        return usercentricsSDK.denyAllForTCF(
            fromLayer = fromLayer ?: TCFDecisionUILayer.FIRST_LAYER,
            consentType = UsercentricsConsentType.EXPLICIT
        )
    }

    override fun save(fromLayer: TCFDecisionUILayer, userDecisions: List<PredefinedUIDecision>): PredefinedUIResponse {
        assert(userDecisions.isNotEmpty()) { "Failed while trying to save decisions: userDecisions parameter cannot be empty" }

        val consents = when (variant) {
            UsercentricsVariant.TCF -> saveDecisionsTCF(userDecisions, fromLayer)
            UsercentricsVariant.DEFAULT -> saveDecisionsDefault(userDecisions)
            UsercentricsVariant.CCPA -> {
                assert(false) { "Failed while trying to save decisions for CCPA: this variant does not allow granular decisions" }
                saveDecisionsDefault(userDecisions)
            }
        }

        val eventType = if (fromLayer == TCFDecisionUILayer.FIRST_LAYER) {
            UsercentricsAnalyticsEventType.SAVE_FIRST_LAYER
        } else {
            UsercentricsAnalyticsEventType.SAVE_SECOND_LAYER
        }
        trackAnalyticsEvent(eventType)

        return PredefinedUIResponse(
            userInteraction = PredefinedUIInteraction.GRANULAR,
            consents = consents,
            controllerId = controllerId,
        )
    }

    private fun saveDecisionsDefault(userDecisions: List<PredefinedUIDecision>): List<UsercentricsServiceConsent> {
        val userDecisionsForDefault = ServicesIdStrategy.userDecisionsGDPR(userDecisions)
        return usercentricsSDK.saveDecisions(userDecisionsForDefault, UsercentricsConsentType.EXPLICIT)
    }

    private fun saveDecisionsTCF(userDecisions: List<PredefinedUIDecision>, fromLayer: TCFDecisionUILayer?): List<UsercentricsServiceConsent> {
        assert(fromLayer != null) { "Failed while trying to save TCF decisions: fromLayer parameter cannot be null" }
        return usercentricsSDK.saveDecisionsForTCF(
            tcfDecisions = ServicesIdStrategy.userDecisionsTCF(userDecisions),
            serviceDecisions = ServicesIdStrategy.userDecisionsGDPR(userDecisions),
            fromLayer = fromLayer ?: TCFDecisionUILayer.FIRST_LAYER,
            consentType = UsercentricsConsentType.EXPLICIT
        )
    }

    override fun close(): PredefinedUIResponse {
        return PredefinedUIResponse(
            userInteraction = PredefinedUIInteraction.NO_INTERACTION,
            consents = usercentricsSDK.getConsents(),
            controllerId = controllerId,
        )
    }

    private fun trackAnalyticsEvent(eventType: UsercentricsAnalyticsEventType) {
        usercentricsSDK.track(eventType)
    }
}
