package com.usercentrics.sdk.services.dataFacade

import com.usercentrics.sdk.acm.service.AdditionalConsentModeService
import com.usercentrics.sdk.assert
import com.usercentrics.sdk.errors.NotReadyException
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.extensions.secondsToMillis
import com.usercentrics.sdk.isTVOS
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.common.UsercentricsVariant
import com.usercentrics.sdk.models.dataFacade.MergedAndUpdatedServicesPair
import com.usercentrics.sdk.models.dataFacade.MergedServicesSettings
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.services.deviceStorage.models.StorageSettings
import com.usercentrics.sdk.services.settings.ISettingsLegacy
import com.usercentrics.sdk.services.tcf.TCFUseCase
import com.usercentrics.sdk.v2.consent.data.ConsentStatus
import com.usercentrics.sdk.v2.consent.data.DataTransferObject
import com.usercentrics.sdk.v2.consent.service.ConsentsService
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.settings.service.ISettingsService

/**
 * Class for DataFacade
 * @suppress
 * */
internal class DataFacade(
    private val consentsService: ConsentsService,
    private val settingsInstance: ISettingsLegacy,
    private val settingsService: ISettingsService,
    private val storageInstance: DeviceStorage,
    private val tcfInstance: TCFUseCase,
    private val additionalConsentModeService: AdditionalConsentModeService,
    private val logger: UsercentricsLogger
) {

    companion object {
        val maxStorageHistorySize = if (isTVOS) 1 else 3
    }

    private fun getSettings(): UsercentricsSettings? = settingsService.settings?.data

    fun execute(
        controllerId: String,
        services: List<LegacyService>,
        consentAction: UsercentricsConsentAction,
        consentType: UsercentricsConsentType,
    ) {
        val settings = getSettings()
        assertSettings(settings)
        settings ?: return

        val dataTransferObject = DataTransferObject.create(
            settings = settings,
            controllerId = settingsInstance.getSettings().controllerId,
            services = services,
            consentAction = consentAction,
            consentType = consentType
        )

        val servicesWithHistory = appendConsentsToHistory(controllerId, services, dataTransferObject)

        val newServices = run {
            val mergedServices = settingsInstance.getSettings().services.updateServices(servicesWithHistory)
            ensureServicesHistorySize(mergedServices)
        }
        val newSettings = settingsInstance.getSettings().copy(services = newServices)
        settingsInstance.setSettings(newSettings)

        storageInstance.saveSettings(settingsInstance.getSettings(), newServices)

        consentsService.saveConsentsState(consentAction)

        if (consentAction != UsercentricsConsentAction.INITIAL_PAGE_LOAD) {
            storageInstance.clearUserActionRequired()
        }
    }

    private fun ensureServicesHistorySize(services: List<LegacyService>): List<LegacyService> {
        return services.map { service ->
            if (service.consent.history.size > maxStorageHistorySize) {
                service.copy(
                    consent = service.consent.copy(
                        history = service.consent.history.takeLast(maxStorageHistorySize)
                    )
                )
            } else {
                service
            }
        }
    }

    fun restoreUserSession(
        controllerId: String,
        activeVariant: UsercentricsVariant?,
        onSuccess: () -> Unit, onError: (UsercentricsException) -> Unit
    ) {
        val settings = getSettings()
        assertSettings(settings)

        if (settings == null) {
            onError(NotReadyException())
            return
        }

        consentsService.getRemoteUserConsents(
            controllerId,
            onSuccess = { consentsData ->
                // GDPR
                val consents = consentsData.consents
                val consentsWithoutRestoredSessions = removeRestoredSessionEvents(consents)
                if (consentsWithoutRestoredSessions.isNotEmpty()) {
                    restoreServicesConsents(controllerId = controllerId, settings = settings, consentsWithoutRestoredSessions = consentsWithoutRestoredSessions)
                } else {
                    logger.debug("No services consents have been restored for $controllerId")
                }

                // TCF
                if (activeVariant == UsercentricsVariant.TCF) {
                    val acString = consentsData.acString
                    if (settingsInstance.isAdditionalConsentModeEnabled()) {
                        additionalConsentModeService.save(acString)
                    }

                    val consentStringObject = consentsData.consentStringObject
                    if (consentStringObject != null) {
                        tcfInstance.restore(consentStringObject.string, acString, consentStringObject.tcfVendorsDisclosedMap)
                    } else {
                        logger.debug("No consentString data, it is needed to restore the TCF session")
                    }
                }

                onSuccess()
            },
            onError = {
                logger.error("Failed while restoring user session", it)
                onError(UsercentricsException(it.toString(), it))
            }
        )
    }

    private fun restoreServicesConsents(
        controllerId: String,
        consentsWithoutRestoredSessions: List<ConsentStatus>,
        settings: UsercentricsSettings,
    ) {
        val storedSettings = settingsInstance.getSettings()
        val storedServices: MutableList<LegacyService> = storedSettings.services.toMutableList()

        val dataTransferObjects = mutableListOf<DataTransferObject>()
        val updatedServices = mutableMapOf<String, LegacyService>()

        val filteredTimestamps = mutableListOf<Long>()
        var lastInteractionTimestampFromRestoredConsents = 0L

        consentsWithoutRestoredSessions.forEach { consent ->
            val consentTimestampMillis = consent.timestampInSeconds.secondsToMillis()
            if (consent.timestampInSeconds in filteredTimestamps) {
                return@forEach
            }

            // Filter repeated consents
            val actionConsents = consentsWithoutRestoredSessions.filter {
                consent.timestampInSeconds == it.timestampInSeconds && it.action == consent.action
            }

            if (actionConsents.isEmpty()) {
                return@forEach
            }
            val actionServices = mutableListOf<LegacyService>()

            actionConsents.forEach actionConsentsLoop@{ actionConsent ->
                val matchingServiceIndex = storedServices.indexOfFirst { service ->
                    service.id == actionConsent.consentTemplateId
                }
                if (matchingServiceIndex < 0) {
                    return@actionConsentsLoop
                }

                val matchingService = storedServices[matchingServiceIndex]

                val updatedService = LegacyService(
                    dataCollected = matchingService.dataCollected,
                    dataDistribution = matchingService.dataDistribution,
                    dataPurposes = matchingService.dataPurposes,
                    dataRecipients = matchingService.dataRecipients,
                    serviceDescription = matchingService.serviceDescription,
                    id = matchingService.id,
                    legalBasis = matchingService.legalBasis,
                    name = matchingService.name,
                    processingCompany = matchingService.processingCompany,
                    retentionPeriodDescription = matchingService.retentionPeriodDescription,
                    technologiesUsed = matchingService.technologiesUsed,
                    urls = matchingService.urls,
                    version = matchingService.version,
                    categorySlug = matchingService.categorySlug,
                    categoryLabel = matchingService.categoryLabel,
                    isEssential = matchingService.isEssential,
                    processorId = matchingService.processorId,
                    subServices = matchingService.subServices,
                    consent = LegacyConsent(
                        history = matchingService.consent.history.takeLast(maxStorageHistorySize),
                        status = actionConsent.consentStatus
                    ),
                    cookieMaxAgeSeconds = matchingService.cookieMaxAgeSeconds,
                    usesNonCookieAccess = matchingService.usesNonCookieAccess,
                    deviceStorageDisclosureUrl = matchingService.deviceStorageDisclosureUrl,
                    deviceStorage = matchingService.deviceStorage,
                    disableLegalBasis = matchingService.disableLegalBasis,
                    isHidden = matchingService.isHidden,
                    defaultConsentStatus = matchingService.defaultConsentStatus,
                )

                updatedServices[matchingService.id] = updatedService
                storedServices[matchingServiceIndex] = updatedService
                actionServices.add(updatedService)
            }

            filteredTimestamps.add(consent.timestampInSeconds)

            val consentAction = consent.action?.let { UsercentricsConsentAction.from(it) } ?: return@forEach
            dataTransferObjects.add(
                DataTransferObject.create(
                    controllerId = controllerId,
                    settings = settings,
                    services = actionServices,
                    consentAction = consentAction,
                    consentType = consentAction.getType(),
                    timestampInMillis = consentTimestampMillis
                )
            )

            lastInteractionTimestampFromRestoredConsents = if (consentTimestampMillis > lastInteractionTimestampFromRestoredConsents) {
                consentTimestampMillis
            } else {
                lastInteractionTimestampFromRestoredConsents
            }
        }

        val updatedServicesWithClearedHistory: List<LegacyService> = storedServices.map { service ->
            val shouldClearHistory = updatedServices.containsKey(service.id)
            if (shouldClearHistory) {
                service.copy(consent = service.consent.copy(history = emptyList()))
            } else {
                service
            }
        }

        var servicesWithMergedHistoryFromRestore = updatedServicesWithClearedHistory
        dataTransferObjects.forEach { dto ->
            servicesWithMergedHistoryFromRestore = appendConsentsToHistory(controllerId, servicesWithMergedHistoryFromRestore, dto)
        }

        val versionInRestoredConsents = consentsWithoutRestoredSessions[consentsWithoutRestoredSessions.size - 1].settingsVersion

        // By doing this, we can assure that the version that is restored, if is not the most recent compared
        // with the one on the current configurations, then a resurface can occur
        val versionToBeSaved = if (versionInRestoredConsents <= storedSettings.version) {
            versionInRestoredConsents
        } else {
            storedSettings.version
        }

        val versionsMatched = versionToBeSaved == versionInRestoredConsents

        val newSettings = storedSettings.copy(
            controllerId = controllerId,
            version = versionToBeSaved,
            services = storedSettings.services.updateServices(servicesWithMergedHistoryFromRestore),
            restoredSessionLastInteractionTimestamp = if (versionsMatched) {
                lastInteractionTimestampFromRestoredConsents
            } else {
                null
            },
        )

        storageInstance.saveSettings(newSettings, servicesWithMergedHistoryFromRestore)
        settingsInstance.setSettings(newSettings)
    }

    fun getMergedServicesAndSettingsFromStorage(): MergedServicesSettings {
        val storageSettings = storageInstance.fetchSettings()
        val essentialServices = getMergedAndUpdatedEssentialServices(storageSettings)
        val mergedAndUpdatedNonEssentialServicesPair = getMergedAndUpdatedNonEssentialServices(storageSettings)

        val mergedServices = mutableListOf<LegacyService>()
        mergedServices.addAll(essentialServices.mergedServices)
        mergedServices.addAll(mergedAndUpdatedNonEssentialServicesPair.mergedServices)
        mergedServices.addAll(mergedAndUpdatedNonEssentialServicesPair.updatedServices)

        val settings = settingsInstance.getSettings()

        return MergedServicesSettings(
            mergedServices = mergedServices,
            mergedSettings = settings.copy(
                controllerId = storageSettings.controllerId.ifBlank { settings.controllerId },
            ),
            updatedEssentialServices = essentialServices.updatedServices,
            updatedNonEssentialServices = mergedAndUpdatedNonEssentialServicesPair.updatedServices,
        )
    }


    fun mergeSettingsFromStorage(controllerId: String, shouldAcceptAllImplicitlyOnInit: Boolean): MergedServicesSettings? {
        val settings = getSettings()
        assertSettings(settings)
        settings ?: return null

        val mergedServicesSettings = getMergedServicesAndSettingsFromStorage()
        val mergedServices = mergedServicesSettings.mergedServices
        val mergedSettings = mergedServicesSettings.mergedSettings
        val updatedEssentialServices = mergedServicesSettings.updatedEssentialServices
        val updatedNonEssentialServices = mergedServicesSettings.updatedNonEssentialServices

        var mergedServicesWithHistory: List<LegacyService> = mergedServices
        val essentialServicesChanges = updatedEssentialServices.isNotEmpty()
        if (essentialServicesChanges) {
            val dataTransferObject = DataTransferObject.create(
                controllerId = mergedSettings.controllerId,
                settings = settings,
                services = updatedEssentialServices,
                consentAction = UsercentricsConsentAction.ESSENTIAL_CHANGE,
                consentType = UsercentricsConsentType.IMPLICIT,
            )
            mergedServicesWithHistory = appendConsentsToHistory(controllerId, mergedServices, dataTransferObject)
        }

        if (updatedNonEssentialServices.isNotEmpty() && !shouldAcceptAllImplicitlyOnInit) {
            val dataTransferObject = DataTransferObject.create(
                controllerId = mergedSettings.controllerId,
                settings = settings,
                services = updatedNonEssentialServices,
                consentAction = UsercentricsConsentAction.INITIAL_PAGE_LOAD,
                consentType = UsercentricsConsentType.IMPLICIT
            )
            mergedServicesWithHistory = appendConsentsToHistory(controllerId, mergedServices, dataTransferObject)
        }

        val newSettings = mergedSettings.copy(services = settingsInstance.getSettings().services.updateServices(mergedServicesWithHistory))
        settingsInstance.setSettings(newSettings)
        storageInstance.saveSettings(newSettings, mergedServicesWithHistory)

        if (essentialServicesChanges) {
            consentsService.saveConsentsState(UsercentricsConsentAction.ESSENTIAL_CHANGE)
        }

        return mergedServicesSettings
    }

    private fun getMergedAndUpdatedEssentialServices(storageSettings: StorageSettings): MergedAndUpdatedServicesPair {
        val services = settingsInstance.getSettings().services.filter { it.isEssential }.sortByName()
        val updatedServices = mutableListOf<LegacyService>()

        val mergedServices = services.map { service ->
            val matchingServiceFromStorage = storageSettings.services.find { storageService ->
                storageService.id == service.id
            }

            if (matchingServiceFromStorage != null) {
                val updatedService = LegacyService(
                    dataCollected = service.dataCollected,
                    dataDistribution = service.dataDistribution,
                    dataPurposes = service.dataPurposes,
                    dataRecipients = service.dataRecipients,
                    serviceDescription = service.serviceDescription,
                    id = service.id,
                    legalBasis = service.legalBasis,
                    name = service.name,
                    processingCompany = service.processingCompany,
                    retentionPeriodDescription = service.retentionPeriodDescription,
                    technologiesUsed = service.technologiesUsed,
                    urls = service.urls,
                    version = service.version,
                    categorySlug = service.categorySlug,
                    categoryLabel = service.categoryLabel,
                    isEssential = service.isEssential,
                    subServices = service.subServices,
                    processorId = matchingServiceFromStorage.processorId,
                    consent = LegacyConsent(
                        history = matchingServiceFromStorage.history.map { it.toConsentHistory() }
                            .takeLast(maxStorageHistorySize),
                        status = true
                    ),
                    cookieMaxAgeSeconds = service.cookieMaxAgeSeconds,
                    usesNonCookieAccess = service.usesNonCookieAccess,
                    deviceStorageDisclosureUrl = service.deviceStorageDisclosureUrl,
                    deviceStorage = service.deviceStorage,
                    disableLegalBasis = service.disableLegalBasis,
                    isHidden = service.isHidden,
                    defaultConsentStatus = service.defaultConsentStatus,
                )

                if (!matchingServiceFromStorage.status) {
                    updatedServices.add(updatedService)
                }

                return@map updatedService
            } else {
                return@map service
            }
        }

        return MergedAndUpdatedServicesPair(
            mergedServices = mergedServices,
            updatedServices = updatedServices
        )
    }

    private fun getMergedAndUpdatedNonEssentialServices(storageSettings: StorageSettings): MergedAndUpdatedServicesPair {
        val services = settingsInstance.getSettings().services.filter { !it.isEssential }.sortByName()

        val mergedServices = mutableListOf<LegacyService>()
        val updatedServices = mutableListOf<LegacyService>()

        services.forEach { service ->
            val matchingServiceFromStorage = storageSettings.services.find { storageService ->
                storageService.id == service.id
            }

            if (matchingServiceFromStorage == null) {
                updatedServices.add(service)
                return@forEach
            }

            mergedServices.add(
                LegacyService(
                    dataCollected = service.dataCollected,
                    dataDistribution = service.dataDistribution,
                    dataPurposes = service.dataPurposes,
                    dataRecipients = service.dataRecipients,
                    serviceDescription = service.serviceDescription,
                    id = service.id,
                    legalBasis = service.legalBasis,
                    name = service.name,
                    processingCompany = service.processingCompany,
                    retentionPeriodDescription = service.retentionPeriodDescription,
                    technologiesUsed = service.technologiesUsed,
                    urls = service.urls,
                    version = service.version,
                    categorySlug = service.categorySlug,
                    categoryLabel = service.categoryLabel,
                    isEssential = service.isEssential,
                    subServices = service.subServices,
                    processorId = matchingServiceFromStorage.processorId,
                    consent = LegacyConsent(
                        history = matchingServiceFromStorage.history.map { it.toConsentHistory() }.takeLast(maxStorageHistorySize),
                        status = matchingServiceFromStorage.status
                    ),
                    cookieMaxAgeSeconds = service.cookieMaxAgeSeconds,
                    usesNonCookieAccess = service.usesNonCookieAccess,
                    deviceStorageDisclosureUrl = service.deviceStorageDisclosureUrl,
                    deviceStorage = service.deviceStorage,
                    disableLegalBasis = service.disableLegalBasis,
                    isHidden = service.isHidden,
                    defaultConsentStatus = service.defaultConsentStatus,
                )
            )
        }
        return MergedAndUpdatedServicesPair(mergedServices, updatedServices)
    }

    private fun appendConsentsToHistory(
        controllerId: String,
        services: List<LegacyService>,
        dataTransferObject: DataTransferObject
    ): List<LegacyService> {
        return services.map { service ->
            run {
                val matchingServiceIndex = dataTransferObject.services.indexOfFirst { dtoService ->
                    dtoService.id == service.id
                }

                val storageServices = storageInstance.fetchSettings().services
                val matchingStorageService = storageServices.find { storageService ->
                    storageService.id == service.id
                }

                if (matchingServiceIndex > -1) {
                    val historyArray = mutableListOf<LegacyConsentHistoryEntry>()
                    historyArray.addAll(service.consent.history)
                    historyArray.add(
                        mapConsentHistoryObject(
                            dataTransferObject,
                            matchingServiceIndex
                        )
                    )

                    val latestHistoryConsent = historyArray[historyArray.lastIndex]

                    if (controllerId == storageInstance.getControllerId() && matchingStorageService != null) {
                        val settingsTimeInMillis = latestHistoryConsent.timestampInMillis
                        var storageTimeInMillis = 0L

                        if (matchingStorageService.history.isNotEmpty()) {
                            storageTimeInMillis = matchingStorageService
                                .history[matchingStorageService.history.lastIndex]
                                .timestampInMillis
                        }

                        if (storageTimeInMillis >= settingsTimeInMillis) {
                            return@map LegacyService(
                                dataCollected = service.dataCollected,
                                dataDistribution = service.dataDistribution,
                                dataPurposes = service.dataPurposes,
                                dataRecipients = service.dataRecipients,
                                serviceDescription = service.serviceDescription,
                                id = service.id,
                                legalBasis = service.legalBasis,
                                name = service.name,
                                processingCompany = service.processingCompany,
                                retentionPeriodDescription = service.retentionPeriodDescription,
                                technologiesUsed = service.technologiesUsed,
                                urls = service.urls,
                                version = service.version,
                                categorySlug = service.categorySlug,
                                categoryLabel = service.categoryLabel,
                                isEssential = service.isEssential,
                                processorId = service.processorId,
                                subServices = service.subServices,
                                consent = LegacyConsent(
                                    status = matchingStorageService.status,
                                    history = matchingStorageService.history.map { it.toConsentHistory() }
                                        .takeLast(maxStorageHistorySize)
                                ),
                                cookieMaxAgeSeconds = service.cookieMaxAgeSeconds,
                                usesNonCookieAccess = service.usesNonCookieAccess,
                                deviceStorageDisclosureUrl = service.deviceStorageDisclosureUrl,
                                deviceStorage = service.deviceStorage,
                                disableLegalBasis = service.disableLegalBasis,
                                isHidden = service.isHidden,
                                defaultConsentStatus = service.defaultConsentStatus,
                            )
                        }
                    }

                    return@map LegacyService(
                        dataCollected = service.dataCollected,
                        dataDistribution = service.dataDistribution,
                        dataPurposes = service.dataPurposes,
                        dataRecipients = service.dataRecipients,
                        serviceDescription = service.serviceDescription,
                        id = service.id,
                        legalBasis = service.legalBasis,
                        name = service.name,
                        processingCompany = service.processingCompany,
                        retentionPeriodDescription = service.retentionPeriodDescription,
                        technologiesUsed = service.technologiesUsed,
                        urls = service.urls,
                        version = service.version,
                        categorySlug = service.categorySlug,
                        categoryLabel = service.categoryLabel,
                        isEssential = service.isEssential,
                        processorId = service.processorId,
                        subServices = service.subServices,
                        consent = LegacyConsent(
                            status = latestHistoryConsent.status,
                            history = historyArray.takeLast(maxStorageHistorySize)
                        ),
                        cookieMaxAgeSeconds = service.cookieMaxAgeSeconds,
                        usesNonCookieAccess = service.usesNonCookieAccess,
                        deviceStorageDisclosureUrl = service.deviceStorageDisclosureUrl,
                        deviceStorage = service.deviceStorage,
                        disableLegalBasis = service.disableLegalBasis,
                        isHidden = service.isHidden,
                        defaultConsentStatus = service.defaultConsentStatus,
                    )
                }

                return@map service
            }
        }
    }

    private fun mapConsentHistoryObject(dataTransferObject: DataTransferObject, serviceIndex: Int): LegacyConsentHistoryEntry {
        val timestampInSeconds = dataTransferObject.timestampInSeconds
        val timestampInMillis = timestampInSeconds.secondsToMillis()

        return LegacyConsentHistoryEntry(
            action = dataTransferObject.consent.action,
            status = dataTransferObject.services[serviceIndex].status,
            type = dataTransferObject.consent.type,
            language = dataTransferObject.settings.language,
            timestampInMillis = timestampInMillis
        )
    }

    private fun removeRestoredSessionEvents(consents: List<ConsentStatus>): List<ConsentStatus> {
        return consents.filter {
            it.action != UsercentricsConsentAction.SESSION_RESTORED.text
        }
    }

    private fun assertSettings(settings: UsercentricsSettings?) = assert(settings != null) { "Illegal settings state (null)" }

}
