package com.usercentrics.sdk.services.deviceStorage.migrations

import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.services.dataFacade.DataFacade
import com.usercentrics.sdk.services.deviceStorage.StorageHolder
import com.usercentrics.sdk.services.deviceStorage.StorageKeys
import com.usercentrics.sdk.services.deviceStorage.models.StorageSettings

internal class MigrationToVersion3(
    storageHolder: StorageHolder,
    private val json: JsonParser,
    private val isTVOS: Boolean
) : AbstractMigrationSettingsV2(3, storageHolder, json) {

    override fun migrate() {
        if (!isTVOS) {
            // Migration not needed
            return
        }

        val settingsValue = storageHolder.usercentricsKeyValueStorage.getString("settings", null)
        if (settingsValue.isNullOrBlank()) {
            // Migration not needed
            return
        }

        // Preserve important data
        val storageVersionValue = storageHolder.usercentricsKeyValueStorage.getNumber("storage_version", -1)
        val ccpaTimestampValue = storageHolder.usercentricsKeyValueStorage.getString("ccpa_timestamp_millis", null)
        val consentsBufferValue = storageHolder.usercentricsKeyValueStorage.getString("consents_buffer", null)
        val sessionTimestampValue = storageHolder.usercentricsKeyValueStorage.getString("session_timestamp", null)
        val tcfValue = storageHolder.usercentricsKeyValueStorage.getString("tcf", null)

        storageHolder.usercentricsKeyValueStorage.purgeStorage()

        val newSettings = processSettings(storageSettingsFromCache(settingsValue))

        storageHolder.usercentricsKeyValueStorage.put("settings", json.encodeToString(StorageSettings.serializer(), newSettings))

        // Restore important data
        if (storageVersionValue != -1) {
            storageHolder.usercentricsKeyValueStorage.put("storage_version", storageVersionValue)
        }

        if (ccpaTimestampValue != null) {
            storageHolder.usercentricsKeyValueStorage.put("ccpa_timestamp_millis", ccpaTimestampValue)
        }

        if (consentsBufferValue != null) {
            storageHolder.usercentricsKeyValueStorage.put("consents_buffer", consentsBufferValue)
        }

        if (sessionTimestampValue != null) {
            storageHolder.usercentricsKeyValueStorage.put("session_timestamp", sessionTimestampValue)
        }

        if (tcfValue != null) {
            storageHolder.usercentricsKeyValueStorage.put("tcf", tcfValue)
        }
    }

    private fun processSettings(settings: StorageSettings): StorageSettings {
        val servicesWithHistorySize = settings.services.map {
            return@map if (it.history.size > DataFacade.maxStorageHistorySize) {
                it.copy(history = it.history.takeLast(DataFacade.maxStorageHistorySize))
            } else {
                it
            }
        }
        return settings.copy(services = servicesWithHistorySize)
    }

}
