package com.usercentrics.sdk.services.deviceStorage.migrations

import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.services.deviceStorage.StorageHolder
import com.usercentrics.sdk.v2.etag.cache.EtagCacheStorage
import com.usercentrics.sdk.v2.file.IFileStorage
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonPrimitive

internal class MigrationToVersion8(
    storageHolder: StorageHolder,
    private val jsonParser: JsonParser,
    private val fileStorage: IFileStorage,
) : Migration(storageHolder, 8) {

    private val dirsRequiringLanguageCodeAddition = listOf("settings", "aggregator", "translations", "tcf-declarations")

    private val cacheMaxAge
        get() = DateTime().addSeconds(1800).timestamp()

    private val newFileSuffix
        get() = "${EtagCacheStorage.cacheControlSeparator}$cacheMaxAge"

    override fun migrate() {
        try {
            val settings = storageHolder.usercentricsKeyValueStorage.getStringWithKeyStartingWith("settings-")
            if (settings.isEmpty()) {
                return
            }

            val settingsJson = settings.entries.first().value
            if (settingsJson.isBlank()) {
                return
            }

            val settingsJsonObject = jsonParser.decodeFromString<JsonObject>(settingsJson)

            val settingsLanguageValue = settingsJsonObject["language"]?.jsonPrimitive?.content
            if (settingsLanguageValue.isNullOrBlank()) {
                return
            }

            val directoriesStored = fileStorage.ls("")
            if (directoriesStored.isNullOrEmpty()) {
                return
            }

            val etagsDir = directoriesStored.first()

            val etagContentDirs = fileStorage.ls(etagsDir)
            if (etagContentDirs.isNullOrEmpty()) {
                return
            }

            for (dir in etagContentDirs) {
                if (dir in dirsRequiringLanguageCodeAddition) {
                    updateFileWithLanguageCode("$etagsDir/$dir", settingsLanguageValue)
                } else {
                    renameFile("$etagsDir/$dir")
                }
            }
        } catch (e: Exception) {
            fileStorage.rmAll()
        }
    }

    private fun updateFileWithLanguageCode(currentDir: String, languageCode: String) {
        val cachedFileName = fileStorage.ls(currentDir)?.firstOrNull() ?: return

        val newDir = "$currentDir-$languageCode"

        val currentFilePath = "$currentDir/$cachedFileName"
        val newFileName = "$newDir/$cachedFileName$newFileSuffix"

        fileStorage.mkdir(newDir)
        fileStorage.copy(currentFilePath, newFileName)

        fileStorage.rmdir(currentDir)
    }

    private fun renameFile(currentDir: String) {
        val cachedFileName = fileStorage.ls(currentDir)?.firstOrNull() ?: return

        val currentFilePath = "$currentDir/$cachedFileName"
        val newFileName = "$currentDir/$cachedFileName$newFileSuffix"

        fileStorage.copy(currentFilePath, newFileName)

        fileStorage.rm(currentFilePath)
    }
}
