package com.usercentrics.sdk.services.initialValues.variants

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.common.InitialView
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage

private const val RESHOW_CMP_DAYS = 390

internal interface TCFStrategy {

    fun shouldAcceptAllImplicitlyOnInit(gdprAppliesOnTCF: Boolean): Boolean

    fun getInitialView(
        tcfInitialViewOptions: TCFInitialViewOptions
    ): InitialView
}

internal class TCFStrategyImpl(
    private val logger: UsercentricsLogger,
    private val deviceStorage: DeviceStorage
) : TCFStrategy {

    override fun shouldAcceptAllImplicitlyOnInit(gdprAppliesOnTCF: Boolean): Boolean {
        return !gdprAppliesOnTCF
    }

    override fun getInitialView(
        tcfInitialViewOptions: TCFInitialViewOptions
    ): InitialView {

        val tcStringOnPolicyVersion2AndSettingsOnPolicyVersion4 = tcfInitialViewOptions.storedTcStringPolicyVersion == 2 && tcfInitialViewOptions.settingsTCFPolicyVersion == 4
        val noChangesMadeOnVendorsList = !tcfInitialViewOptions.resurfacePurposeChanged && !tcfInitialViewOptions.resurfaceVendorAdded

        val sharedInitialViewOptions = tcfInitialViewOptions.sharedInitialViewOptions

        return when {
            sharedInitialViewOptions.manualResurface -> {
                logger.debug(ShowCMPReasons.manualResurface)
                InitialView.FIRST_LAYER
            }

            tcStringOnPolicyVersion2AndSettingsOnPolicyVersion4 && sharedInitialViewOptions.versionChangeRequiresReshow && noChangesMadeOnVendorsList -> {
                InitialView.NONE
            }

            sharedInitialViewOptions.versionChangeRequiresReshow -> {
                logger.debug(ShowCMPReasons.versionChangeRequiresReshow)
                InitialView.FIRST_LAYER
            }

            tcfInitialViewOptions.noGDPRConsentActionPerformed -> {
                logger.debug(ShowCMPReasons.noGDPRConsentActionPerformed)
                InitialView.FIRST_LAYER
            }

            tcfInitialViewOptions.resurfacePurposeChanged -> {
                logger.debug(ShowCMPReasons.purposeChangedTCF)
                InitialView.FIRST_LAYER
            }

            tcfInitialViewOptions.resurfaceVendorAdded -> {
                logger.debug(ShowCMPReasons.vendorAddedTCF)
                InitialView.FIRST_LAYER
            }

            timeToReshowHasPassed(tcfInitialViewOptions.resurfacePeriodEnded, lastServiceInteractionTimestamp()) -> {
                logger.debug((ShowCMPReasons.periodEndedTCF))
                InitialView.FIRST_LAYER
            }

            tcfInitialViewOptions.resurfaceATPChanged -> {
                logger.debug((ShowCMPReasons.resurfaceATPChanged))
                InitialView.FIRST_LAYER
            }


            else -> {
                InitialView.NONE
            }
        }
    }

    private fun lastServiceInteractionTimestamp(): Long? {
        return deviceStorage.lastInteractionTimestamp()
    }

    private fun timeToReshowHasPassed(reshowCMP: Boolean, lastTimestamp: Long?): Boolean {
        if (!reshowCMP || lastTimestamp == null) {
            return false
        }

        val lastInteractionDateTime = DateTime(lastTimestamp)
        val currentDateTime = DateTime()

        val daysSinceLastInteraction = currentDateTime.diffInDays(lastInteractionDateTime)

        return daysSinceLastInteraction >= RESHOW_CMP_DAYS
    }
}
