package com.usercentrics.sdk.services.settings

import com.usercentrics.sdk.extensions.sortedAlphaBy
import com.usercentrics.sdk.models.api.SERVICE_DEFAULT_DISABLE_LEGAL_BASIS
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.v2.settings.data.*
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization

internal class SettingsServicesMapper(
    private val idsGenerator: IGeneratorIds
): ISettingsServiceMapper {

    override fun map(
        apiSettings: UsercentricsSettings,
        apiServices: List<UsercentricsService>,
        translations: LegalBasisLocalization
    ): List<LegacyService> {

        val categoriesMap = apiSettings.categories?.associateBy {
            it.categorySlug
        } ?: emptyMap()

        val apiSettingsMapped = apiSettings.consentTemplates.map {
            mapService(it, apiServices, translations, categoriesMap)
        }.sortedAlphaBy {
            it.name
        }
        return apiSettingsMapped
    }

    private fun mapService(
        serviceConsentTemplate: ServiceConsentTemplate,
        apiServices: List<UsercentricsService>,
        translations: LegalBasisLocalization,
        categoriesMap: Map<String, UsercentricsCategory>,
    ): LegacyService {
        val category = categoriesMap[serviceConsentTemplate.categorySlug]
        val baseService = mapBaseService(serviceConsentTemplate, apiServices, translations)
        val status = resolveStatus(serviceConsentTemplate, category)
        val processorId = idsGenerator.generateProcessorId()

        return LegacyService(
            dataCollected = baseService.dataCollected,
            dataDistribution = baseService.dataDistribution,
            dataPurposes = baseService.dataPurposes,
            dataRecipients = baseService.dataRecipients,
            serviceDescription = baseService.serviceDescription,
            id = baseService.id,
            legalBasis = baseService.legalBasis,
            name = baseService.name,
            processingCompany = baseService.processingCompany,
            retentionPeriodDescription = baseService.retentionPeriodDescription,
            technologiesUsed = baseService.technologiesUsed,
            urls = baseService.urls,
            version = baseService.version,
            categorySlug = serviceConsentTemplate.categorySlug ?: "",
            categoryLabel = category?.label ?: "",
            consent = LegacyConsent(
                history = listOf(),
                status = status
            ),
            isEssential = category?.isEssential ?: false,
            disableLegalBasis = baseService.disableLegalBasis ?: SERVICE_DEFAULT_DISABLE_LEGAL_BASIS,
            processorId = processorId,
            subServices = mapSubServices(serviceConsentTemplate.subConsents, apiServices, translations),
            cookieMaxAgeSeconds = baseService.cookieMaxAgeSeconds,
            usesNonCookieAccess = baseService.usesNonCookieAccess,
            deviceStorageDisclosureUrl = baseService.deviceStorageDisclosureUrl,
            deviceStorage = baseService.deviceStorage,
            isHidden = baseService.isHidden,
            defaultConsentStatus = serviceConsentTemplate.defaultConsentStatus,
        )
    }

    private fun mapBaseService(serviceConsent: ConsentTemplate, apiServices: List<UsercentricsService>, translations: LegalBasisLocalization): LegacyBasicService {
        val apiAggregatorService = findServicesFromAggregatorArray(serviceConsent, apiServices)

        return LegacyBasicService(
            dataCollected = apiAggregatorService.dataCollectedList,
            dataDistribution = PredefinedUIDataDistribution(
                processingLocation = apiAggregatorService.locationOfProcessing,
                thirdPartyCountries = apiAggregatorService.thirdCountryTransfer
            ),
            dataPurposes = apiAggregatorService.dataPurposesList.ifEmpty { apiAggregatorService.dataPurposes },
            dataRecipients = apiAggregatorService.dataRecipientsList,
            serviceDescription = apiAggregatorService.descriptionOfService,
            id = serviceConsent.templateId,
            legalBasis = resolveLegalBasisList(serviceConsent, apiAggregatorService, translations),
            name = if (!apiAggregatorService.dataProcessor.isNullOrBlank()) {
                apiAggregatorService.dataProcessor
            } else {
                ""
            },
            processingCompany = PredefinedUIProcessingCompany(
                address = apiAggregatorService.addressOfProcessingCompany,
                dataProtectionOfficer = apiAggregatorService.dataProtectionOfficer,
                name = apiAggregatorService.nameOfProcessingCompany
            ),
            retentionPeriodDescription = apiAggregatorService.retentionPeriodDescription,
            technologiesUsed = apiAggregatorService.technologyUsed,
            urls = PredefinedUIURLs(
                cookiePolicy = apiAggregatorService.cookiePolicyURL,
                dataProcessingAgreement = apiAggregatorService.linkToDpa,
                optOut = apiAggregatorService.optOutUrl,
                privacyPolicy = apiAggregatorService.privacyPolicyURL
            ),
            version = serviceConsent.version,
            disableLegalBasis = if (serviceConsent is ServiceConsentTemplate) {
                serviceConsent.disableLegalBasis
            } else {
                false
            },
            cookieMaxAgeSeconds = apiAggregatorService.cookieMaxAgeSeconds,
            usesNonCookieAccess = apiAggregatorService.usesNonCookieAccess,
            deviceStorageDisclosureUrl = apiAggregatorService.deviceStorageDisclosureUrl,
            deviceStorage = apiAggregatorService.deviceStorage,
            dpsDisplayFormat = apiAggregatorService.dpsDisplayFormat,
            isHidden = apiAggregatorService.isHidden
        )
    }

    private fun findServicesFromAggregatorArray(serviceConsent: ConsentTemplate, apiServices: List<UsercentricsService>): UsercentricsService {
        return apiServices.find {
            serviceConsent.templateId == it.templateId && serviceConsent.version == it.version
        } ?: createUsercentricsService(serviceConsent)
    }

    private fun createUsercentricsService(serviceConsent: ConsentTemplate): UsercentricsService {
        return UsercentricsService(
            description = serviceConsent.description,
            templateId = serviceConsent.templateId,
            version = serviceConsent.version,
            isHidden = serviceConsent.isHidden,
            categorySlug = serviceConsent.categorySlug,
            isDeactivated = serviceConsent.isDeactivated
        )
    }

    private fun resolveStatus(apiService: ServiceConsentTemplate, apiCategory: UsercentricsCategory?): Boolean {
        return if (apiCategory?.isEssential == true) {
            true
        } else {
            apiService.defaultConsentStatus ?: false
        }
    }

    private  fun mapSubServices(
        apiSubServices: List<SubConsentTemplate>,
        apiServices: List<UsercentricsService>,
        translations: LegalBasisLocalization
    ): List<LegacyBasicService> {
        return apiSubServices.map { mapBaseService(it, apiServices, translations) }
    }

    private fun resolveLegalBasisList(apiService: ConsentTemplate, apiAggregatorService: UsercentricsService, translations: LegalBasisLocalization): List<String> {
        return when {
            (apiService is ServiceConsentTemplate) && (apiService.legalBasisList?.isNotEmpty() == true) -> {
                apiService.legalBasisList.map {
                    translations.data?.get(it) ?: it
                }
            }

            apiAggregatorService.legalBasisList.isNotEmpty() -> {
                apiAggregatorService.legalBasisList
            }

            else -> {
                listOf(apiAggregatorService.legalGround)
            }
        }
    }
}
