package com.usercentrics.sdk.services.tcf

import com.usercentrics.sdk.acm.service.AdditionalConsentModeService
import com.usercentrics.sdk.assertNotUIThread
import com.usercentrics.sdk.core.settings.SettingsOrchestrator
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.extensions.sortedAlphaBy
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.settings.PredefinedUIDecision
import com.usercentrics.sdk.models.settings.UsercentricsConsentAction
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.services.deviceStorage.models.StorageTCF
import com.usercentrics.sdk.services.deviceStorage.models.StorageVendor
import com.usercentrics.sdk.services.tcf.Constants.PURPOSE_ONE_ID
import com.usercentrics.sdk.services.tcf.interfaces.*
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import com.usercentrics.sdk.v2.async.dispatcher.Semaphore
import com.usercentrics.sdk.v2.consent.service.ConsentsService
import com.usercentrics.sdk.v2.location.service.ILocationService
import com.usercentrics.sdk.v2.settings.data.TCF2ChangedPurposes
import com.usercentrics.sdk.v2.settings.data.TCF2Settings
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.settings.service.ISettingsService
import com.usercentrics.sdk.v2.tcf.facade.TCFFacade
import com.usercentrics.tcf.core.*
import com.usercentrics.tcf.core.encoder.TCFKeysEncoder
import com.usercentrics.tcf.core.model.ConsentLanguages
import com.usercentrics.tcf.core.model.PurposeRestriction
import com.usercentrics.tcf.core.model.RestrictionType
import com.usercentrics.tcf.core.model.gvl.DataRetention
import com.usercentrics.tcf.core.model.gvl.RetentionPeriod
import com.usercentrics.tcf.core.model.gvl.Vendor

@Suppress("SpellCheckingInspection")
internal class TCF(
    private val logger: UsercentricsLogger,
    private val settingsService: ISettingsService,
    private val storageInstance: DeviceStorage,
    private val consentsService: ConsentsService,
    private val locationService: ILocationService,
    private val additionalConsentModeService: AdditionalConsentModeService,
    private val tcfFacade: TCFFacade,
    private val dispatcher: Dispatcher,
    private val semaphore: Semaphore,
    private val settingsOrchestrator: SettingsOrchestrator,
) : TCFUseCase {

    private val vendors: MutableList<TCFVendor> = mutableListOf()
    private val purposes: MutableList<TCFPurpose> = mutableListOf()

    private var tcModel: TCModel? = null
    private var tcfData: TCFData? = null
    private var disclosedVendorsMap: MutableMap<Int, StorageVendor> = mutableMapOf()
    private var changedPurposes: TCF2ChangedPurposes = TCF2ChangedPurposes()

    private val settings: UsercentricsSettings?
        get() = settingsService.settings?.data

    private val tcfSettings: TCF2Settings?
        get() = settings?.tcf2

    private val vendorsIdsWithImplicitLegitimateInterest = mutableListOf<Int>()

    override suspend fun initialize(settingsId: String): Result<Unit> {
        val tcf2Settings = tcfSettings ?: return Result.failure(UsercentricsException("TCF Options are empty", IllegalStateException()))

        val storedTCFData = storageInstance.bootTCFData(settingsId)

        val initTCModelResult = this.initTCModel(tcf2Settings = tcf2Settings, storedTCFData = storedTCFData)
        val exceptionOnInitTCModel = initTCModelResult.exceptionOrNull()
        if (exceptionOnInitTCModel != null) {
            return Result.failure(exceptionOnInitTCModel)
        }

        updateLocalStorageIfNeeded(settingsId = settingsId, actualTCFSettings = storedTCFData)
        return Result.success(Unit)
    }

    override fun restore(tcString: String, acString: String, vendorsDisclosed: Map<Int, StorageVendor>) {
        val tcf2Settings = tcfSettings!!

        // Overwrite storage
        setDisclosedVendors(tcf2Settings, vendorsDisclosed)
        storageInstance.saveTCFData(
            tcfData = StorageTCF(
                vendorsDisclosedMap = vendorsDisclosed,
                tcString = tcString,
                acString = acString,
            )
        )

        applyTCString(tcString)
        updateIABTCFKeys(tcString)

        overrideTCModel(tcf2Settings)
        resetTCFData()
    }

    override fun acceptAllDisclosed(fromLayer: TCFDecisionUILayer) {
        runCatching {
            val internalTCModel = tcModel!!

            val purposeConsents = mutableSetOf<Int>()
            val tempSetVendorConsents = mutableListOf<Int>()
            val tempUnsetVendorConsents = mutableListOf<Int>()

            val allVendorsIds = mutableListOf<Int>()
            val purposeLegitimateInterests = mutableSetOf<Int>()

            val allVendors = getVendors()

            // [MSDK-1723] Customer special request
            val excludedVendors = UsercentricsTCFSettings.excludedVendors

            allVendors.forEach { vendor ->
                if (excludedVendors.contains(vendor.id)) {
                    return@forEach // skip excluded vendors from TCF acceptAll
                }

                if (vendor.purposes.isNotEmpty()) {
                    tempSetVendorConsents.add(vendor.id)
                    purposeConsents.addAll(vendor.purposes.map { purpose -> purpose.id })
                } else {
                    tempUnsetVendorConsents.add(vendor.id)
                }

                allVendorsIds.add(vendor.id)
                purposeLegitimateInterests.addAll(vendor.legitimateInterestPurposes.map { purpuse -> purpuse.id })
            }

            setDisclosedVendors(tcfSettings!!, allVendors.toStorageVendorMap())

            internalTCModel.vendorConsents.set(tempSetVendorConsents)
            internalTCModel.vendorConsents.unset(tempUnsetVendorConsents)

            internalTCModel.vendorLegitimateInterests.set(allVendorsIds)

            val tempUnsetLegitimateInterest = mutableListOf<Int>()
            internalTCModel.vendorLegitimateInterests.unset(tempUnsetLegitimateInterest)

            val purposesFlatlyNotAllowed = this.changedPurposes.notAllowedPurposes
            internalTCModel.purposeConsents.set(filterNotAllowedPurposes(purposeConsents, purposesFlatlyNotAllowed))
            internalTCModel.purposeLegitimateInterests.set(filterNotAllowedPurposes(purposeLegitimateInterests, purposesFlatlyNotAllowed))

            // Need to untoggle manually legitimate interest on purposes/vendors to remove previously given consents
            if (tcfSettings!!.hideLegitimateInterestToggles) {
                internalTCModel.unsetAllVendorLegitimateInterests()
                internalTCModel.unsetAllPurposeLegitimateInterests()
            }

            internalTCModel.specialFeatureOptins.set(getSpecialFeatureIdsFromVendorsAndStacks())

            updateTCString(fromLayer)
        }.onFailure {
            logger.error("Something went wrong with TCF acceptAllDisclosed method: $it", it)
        }
    }

    private fun filterNotAllowedPurposes(consents: MutableSet<Int>, notAllowedPurposes: List<Int>): List<Int> {
        if (notAllowedPurposes.isEmpty()) {
            return consents.toList()
        }
        return consents.filter { !notAllowedPurposes.contains(it) }
    }

    override suspend fun changeLanguage(language: String): Result<Unit> {
        val supportedLanguage = resolveLanguage(language)

        val resetGVLWithLanguageResult = this.resetGVLWithLanguage(language = supportedLanguage)
        val exceptionOnChangingLanguage = resetGVLWithLanguageResult.exceptionOrNull()
        if (exceptionOnChangingLanguage != null) {
            return Result.failure(exceptionOnChangingLanguage)
        }

        this.tcModel?.getGvl()?.narrowVendorsTo(this.getRawSelectedVendorIds())
        this.resetTCFData()

        return Result.success(Unit)
    }

    override fun denyAllDisclosed(fromLayer: TCFDecisionUILayer) {
        runCatching {
            val internalTCModel = tcModel!!

            internalTCModel.unsetAllVendorConsents()
            internalTCModel.unsetAllVendorLegitimateInterests()

            internalTCModel.vendorLegitimateInterests.set(vendorsIdsWithImplicitLegitimateInterest)

            internalTCModel.purposeConsents.unset(getPurposeIdsFromVendorsAndStacks())
            internalTCModel.purposeLegitimateInterests.unset(getPurposeIdsFromVendorsAndStacks())
            internalTCModel.specialFeatureOptins.unset(getSpecialFeatureIdsFromVendorsAndStacks())

            setDisclosedVendors(tcfSettings!!, getVendors().toStorageVendorMap())

            updateTCString(fromLayer)
        }.onFailure {
            logger.error("Something went wrong with TCF denyAllDisclosed method: $it", it)
        }
    }

    override fun getTCFData(): TCFData {
        assertNotUIThread()
        semaphore.acquire()

        try {
            if (this.tcfData == null) {
                this.setTCFData()
            }
        } catch (ex: Exception) {
            throw ex
        } finally {
            semaphore.release()
        }
        return this.tcfData!!
    }

    private fun updateLocalStorageIfNeeded(settingsId: String, actualTCFSettings: StorageTCF) {
        setNewGdprAppliesValue()

        val storedTCFSettingsId = storageInstance.getActualTCFSettingsId()
        if (storedTCFSettingsId.isBlank() || storedTCFSettingsId == settingsId) {
            return
        }

        storageInstance.apply {
            saveActualTCFSettingsId(settingsId)
            clearTCFStorageEntries()
        }

        updateIABTCFKeys(actualTCFSettings.tcString)

        val acString = actualTCFSettings.acString
        if (acString?.isNotBlank() == true) {
            additionalConsentModeService.save(acString)
        }
    }

    // given gdprApplies will have a new value that should respect ruleset.noShow property
    // this method will force a migration to update the value from 1 to 0
    private fun setNewGdprAppliesValue() {
        if (isRulesetMarkedNoShow()) {
            storageInstance.storeValuesDefaultStorage(mapOf(IABTCFKeys.GDPR_APPLIES.key to 0))
        }
    }

    private fun setTCFData() {
        assertNotUIThread()

        this.tcfData = TCFData(
            features = this.getFeaturesFromVendors().sortedAlphaBy { it.name },
            purposes = this.getPurposesFromVendors().toMutableList(),
            specialFeatures = this.getSpecialFeaturesFromVendorsAndStacks().sortedAlphaBy { it.name },
            specialPurposes = this.getSpecialPurposesFromVendors().sortedAlphaBy { it.name },
            stacks = this.getStacks().sortedAlphaBy { it.name },
            vendors = this.getVendors().sortedAlphaBy { it.name },
            tcString = storageInstance.getTCFData().tcString,
            thirdPartyCount = thirdPartyCount()
        )
    }

    private fun thirdPartyCount(): Int {
        val gdprServicesCount = settingsService.settings?.servicesCount ?: 0
        val adTechProvidersCount = additionalConsentModeService.adTechProviderList?.size ?: 0
        return this.vendors.size + gdprServicesCount + adTechProvidersCount
    }

    private suspend fun initTCModel(tcf2Settings: TCF2Settings, storedTCFData: StorageTCF): Result<Unit> {
        val gvl = GVL(tcfFacade)
        this.tcModel = TCModel(gvl)

        // Init TCModel with the stored TCString data
        val tcString = storedTCFData.tcString
        if (tcString.isNotBlank()) {
            applyTCString(tcString)
        }

        // Set/Update the values that comes from the settings
        overrideTCModel(tcf2Settings)

        val gvlInitializeResult = gvl.initialize()
        val exceptionOnGVLInitialize = gvlInitializeResult.exceptionOrNull()
        if (exceptionOnGVLInitialize != null) {
            return Result.failure(exceptionOnGVLInitialize)
        }

        // TODO we can probably do this much more efficiently and load the correct language from the start
        val supportedLanguage = resolveLanguage(settings!!.language)
        val resetGVLWithLanguageResult = this.resetGVLWithLanguage(language = supportedLanguage)

        val resetGVLException = resetGVLWithLanguageResult.exceptionOrNull()
        if (resetGVLException != null) {
            return Result.failure(resetGVLException)
        }

        val selectedVendorIds = this.getRawSelectedVendorIds()
        this.tcModel?.getGvl()?.narrowVendorsTo(selectedVendorIds)
        this.tcModel?.publisherRestrictions?.setGvl(gvl)

        if (tcf2Settings.isServiceSpecific) {
            setChangedPurposes(tcf2Settings)
            applyRemoteRestrictions()
        }

        initDisclosedVendors(tcf2Settings, storedTCFData)

        return Result.success(Unit)
    }

    private fun setChangedPurposes(tcf2Settings: TCF2Settings) {
        tcf2Settings.changedPurposes?.let { remoteChangedPurposes ->
            val hasNotAllowedPurposes = remoteChangedPurposes.notAllowedPurposes.isNotEmpty()

            if (hasNotAllowedPurposes) {
                this.changedPurposes = remoteChangedPurposes
                return@let
            } else {
                this.changedPurposes = remoteChangedPurposes.copy(notAllowedPurposes = UsercentricsTCFSettings.purposesFlatlyNotAllowed)
            }
        }
    }

    private fun clearAlreadyAppliedRestrictionsFromTcString(remoteRestrictions: Set<String>) {
        if (this.tcModel?.publisherRestrictions?.map?.isEmpty() == true) {
            return
        }
        val restrictionsAlreadyInPlace = this.tcModel?.publisherRestrictions?.map?.keys?.toSet() ?: emptySet()

        val outdatedRestrictions = restrictionsAlreadyInPlace - remoteRestrictions
        outdatedRestrictions.forEach { this.tcModel?.publisherRestrictions?.map?.remove(it) }
    }

    private fun initDisclosedVendors(tcf2Settings: TCF2Settings, storedTCFData: StorageTCF) {
        if (storedTCFData.vendorsDisclosedMap.isNotEmpty()) {
            setDisclosedVendors(tcf2Settings, storedTCFData.vendorsDisclosedMap)
        }
    }

    private fun applyTCString(tcString: String) {
        try {
            this.tcModel = TCString.decode(tcString, this.tcModel!!)
        } catch (e: Throwable) {
            logger.error(TCF_WARN_MESSAGES.INIT_TCF_ERROR.message, e)
        }
    }

    private fun overrideTCModel(tcf2Settings: TCF2Settings) {
        this.tcModel?.apply {
            setCmpId(StringOrNumber.Int(tcf2Settings.cmpId))
            setCmpVersion(StringOrNumber.Int(tcf2Settings.cmpVersion))
            setIsServiceSpecific(tcf2Settings.isServiceSpecific)
            setPublisherCountryCode(tcf2Settings.publisherCountryCode)
            setPurposeOneTreatment(tcf2Settings.purposeOneTreatment)
        }
    }

    private fun applyRemoteRestrictions() {
        val remoteRestrictions = getRemoteRestrictions()

        clearAlreadyAppliedRestrictionsFromTcString(remoteRestrictions)

        this.tcModel?.publisherRestrictions?.initTCModelRestrictPurposeToLegalBasis(remoteRestrictions)
    }

    private fun getRemoteRestrictions(): Set<String> {
        val consentRestrictions = changedPurposes.purposes.map { purposeId -> PurposeRestriction(purposeId, RestrictionType.REQUIRE_CONSENT).getHash() }
        val legIntRestrictions = changedPurposes.legIntPurposes.map { purposeId -> PurposeRestriction(purposeId, RestrictionType.REQUIRE_LI).getHash() }
        val notAllowedRestrictions = changedPurposes.notAllowedPurposes.map { purposeId -> PurposeRestriction(purposeId, RestrictionType.NOT_ALLOWED).getHash() }

        return (consentRestrictions + legIntRestrictions + notAllowedRestrictions).toSet()
    }

    private fun getVendorRestrictions(vendorId: Int): List<TCFVendorRestriction> {
        val result = mutableListOf<TCFVendorRestriction>()

        this.tcModel?.publisherRestrictions?.getRestrictions(vendorId)?.forEach { restriction ->
            val purpuseId = restriction.getPurposeId()
            if (purpuseId != null) {
                result.add(TCFVendorRestriction(purpuseId, restriction.restrictionType))
            }
        }
        return result
    }

    private fun getVendors(): List<TCFVendor> {
        if (this.vendors.isEmpty()) {
            this.setVendors()
        }
        return this.vendors.toList()
    }

    private fun setVendors() {
        val globalTCModel: TCModel? = this.tcModel
        val tcf2Settings = tcfSettings!!
        val vendorList = mutableListOf<TCFVendor>()
        val implicitLegitimateInterestVendorsIds = mutableListOf<Int>()

        if (globalTCModel != null) {

            val gvl = globalTCModel.getGvl()
            gvl?.vendors?.forEach { vendorEntry ->

                val vendorId = vendorEntry.key
                val vendor: Vendor = vendorEntry.value

                val vendorLegitimateInterestPurposes = vendor.legIntPurposes.map { id -> IdAndName(id, gvl.purposes?.get(id.toString())?.name ?: "") }
                val allVendorPurposes = vendor.purposes.map { id -> IdAndName(id, gvl.purposes?.get(id.toString())?.name ?: "") }

                val vendorPurposes = if (tcf2Settings.purposeOneTreatment) {
                    allVendorPurposes.filter { purpose -> purpose.id != PURPOSE_ONE_ID }.toMutableList()
                } else {
                    allVendorPurposes
                }

                val restrictions = this.getVendorRestrictions(vendorId.toInt())

                // Clone the current purpose state
                var restrictedLegitimateInterestPurposes = vendorLegitimateInterestPurposes.map { purpose -> IdAndName(purpose.id, purpose.name) }.toMutableList()
                var restrictedPurposes = vendorPurposes.map { purpose -> purpose }.toMutableList()

                restrictions.forEach { restriction ->
                    when (restriction.restrictionType) {
                        RestrictionType.REQUIRE_LI -> {
                            restrictedPurposes = restrictedPurposes.filter { purpose ->
                                if (purpose.id == restriction.purposeId) {
                                    if (vendor.flexiblePurposes.contains(purpose.id)) {
                                        restrictedLegitimateInterestPurposes.add(IdAndName(purpose.id, purpose.name))
                                    }
                                    return@filter false
                                }
                                return@filter true
                            }.toMutableList()
                        }

                        RestrictionType.REQUIRE_CONSENT -> {
                            restrictedLegitimateInterestPurposes = restrictedLegitimateInterestPurposes.filter { purpose ->
                                if (purpose.id == restriction.purposeId) {
                                    if (vendor.flexiblePurposes.contains(purpose.id)) {
                                        restrictedPurposes.add(purpose)
                                    }
                                    return@filter false
                                }
                                return@filter true
                            }.toMutableList()
                        }

                        RestrictionType.NOT_ALLOWED -> {
                            restrictedPurposes = restrictedPurposes.filter { purpose ->
                                purpose.id != restriction.purposeId
                            }.toMutableList()

                            restrictedLegitimateInterestPurposes = restrictedLegitimateInterestPurposes.filter { purpose ->
                                purpose.id != restriction.purposeId
                            }.toMutableList()
                        }
                    }
                }

                val features = vendor.features.map { id -> IdAndName(id, gvl.features?.get(id.toString())?.name ?: "") }
                val flexiblePurposes = vendor.flexiblePurposes.map { id -> IdAndName(id, gvl.purposes?.get(id.toString())?.name ?: "") }
                val filteredSpecialFeatures = vendor.specialFeatures.filter { specialFeatureId ->
                    !tcf2Settings.disabledSpecialFeatures.contains(specialFeatureId)
                }

                val specialFeatures = filteredSpecialFeatures.map { id -> IdAndName(id, gvl.specialFeatures?.get(id.toString())?.name ?: "") }
                val specialPurposes = vendor.specialPurposes.map { id -> IdAndName(id, gvl.specialPurposes?.get(id.toString())?.name ?: "") }

                val matchedDataCategories = vendor.dataDeclaration?.map { id -> IdAndName(id, gvl.dataCategories?.get(id.toString())?.name ?: "") }

                val gvlDataRetention = vendor.dataRetention

                if (hasImplicitLegitimateInterest(vendor.purposes, vendor.legIntPurposes, vendor.specialPurposes)) {
                    implicitLegitimateInterestVendorsIds.add(vendor.id)
                }

                vendorList.add(
                    TCFVendor(
                        consent = globalTCModel.vendorConsents.has(vendor.id),
                        cookieMaxAgeSeconds = vendor.cookieMaxAgeSeconds,
                        deviceStorageDisclosureUrl = vendor.deviceStorageDisclosureUrl,
                        features = features,
                        flexiblePurposes = flexiblePurposes,
                        id = vendor.id,
                        legitimateInterestConsent = checklegitimateInterestConsent(vendor, globalTCModel),
                        legitimateInterestPurposes = restrictedLegitimateInterestPurposes,
                        name = vendor.name,
                        policyUrl = vendor.policyUrl,
                        purposes = restrictedPurposes,
                        restrictions = restrictions,
                        showConsentToggle = restrictedPurposes.isNotEmpty() && tcf2Settings.useGranularChoice,
                        showLegitimateInterestToggle = restrictedLegitimateInterestPurposes.isNotEmpty() && tcf2Settings.useGranularChoice && !tcf2Settings.hideLegitimateInterestToggles,
                        specialFeatures = specialFeatures,
                        specialPurposes = specialPurposes,
                        usesNonCookieAccess = vendor.usesNonCookieAccess,
                        usesCookies = vendor.usesCookies,
                        cookieRefresh = vendor.cookieRefresh,
                        dataSharedOutsideEU = tcf2Settings.vendorIdsOutsideEUList.contains(vendor.id),
                        dataRetention = DataRetention(
                            gvlDataRetention?.stdRetention,
                            purposes = RetentionPeriod.parseFromGvlMap(gvlDataRetention?.purposes),
                            specialPurposes = RetentionPeriod.parseFromGvlMap(gvlDataRetention?.specialPurposes),
                        ),
                        dataCategories = matchedDataCategories ?: emptyList(),
                        vendorUrls = vendor.urls ?: emptyList(),
                    )
                )
            }
        }

        this.vendors.apply {
            clear()
            addAll(vendorList.sortedAlphaBy { it.name })
        }

        this.vendorsIdsWithImplicitLegitimateInterest.apply {
            clear()
            addAll(implicitLegitimateInterestVendorsIds)
        }
    }

    private fun checklegitimateInterestConsent(
        vendor: Vendor,
        globalTCModel: TCModel
    ): Boolean? {

        val hasStoredLegitimateInterest = disclosedVendorsMap[vendor.id] != null

        return if (hasStoredLegitimateInterest) {
            val isLegitimateInterestDeclaredInGvl = globalTCModel.vendorLegitimateInterests.has(vendor.id)
            return isLegitimateInterestDeclaredInGvl || hasImplicitLegitimateInterest(vendor.purposes, vendor.legIntPurposes, vendor.specialPurposes)
        } else null
    }

    private fun <T> hasImplicitLegitimateInterest(purposes: List<T>, legIntPurposes: List<T>, specialPurposes: List<T>): Boolean {
        return when {
            purposes.isEmpty() && legIntPurposes.isEmpty() && specialPurposes.isNotEmpty() -> true
            purposes.isNotEmpty() && legIntPurposes.isEmpty() && specialPurposes.isNotEmpty() -> true
            else -> false
        }
    }

    private fun getStacks(): List<TCFStack> {
        val gvl = this.tcModel?.getGvl()
        val disabledSpecialFeatures = tcfSettings!!.disabledSpecialFeatures
        val stacks: MutableList<TCFStack> = mutableListOf()

        if (gvl != null) {
            tcfSettings!!.selectedStacks.forEach { stackId ->
                val stack = gvl.stacks?.get(stackId.toString())

                if (stack != null) {
                    stacks.add(
                        TCFStack(
                            description = stack.description,
                            id = stack.id,
                            name = stack.name,
                            purposeIds = stack.purposes,
                            specialFeatureIds = stack.specialFeatures.filter { specialFeatureId ->
                                !disabledSpecialFeatures.contains(specialFeatureId)
                            }
                        )
                    )
                }
            }
        }
        return stacks
    }

    // Warning: This list could contain IDs that doesn't exist anymore, it is not curated
    private fun getRawSelectedVendorIds(): List<Int> = tcfSettings!!.selectedVendorIds

    private fun getSelectedTCFVendors(): List<TCFVendor> {
        val selectedVendorIds = tcfSettings!!.selectedVendorIds.toSet()
        return getVendors().filter { selectedVendorIds.contains(it.id) }
    }

    private fun setDisclosedVendors(tcf2Settings: TCF2Settings, disclosedVendors: Map<Int, StorageVendor>) {
        disclosedVendorsMap.apply {
            // TODO: ATOMIC?
            clear()
            putAll(disclosedVendors)
        }

        if (!tcf2Settings.isServiceSpecific) {
            this.tcModel!!.vendorsDisclosed.set(disclosedVendors.keys.toList())
        }
    }

    override fun getSettingsTCFPolicyVersion(): Int {
        return Constants.TCF_POLICY_VERSION
    }

    override fun getStoredTcStringPolicyVersion(): Int {
        return this.tcModel!!.getPolicyVersion()
    }

    override fun getHideNonIabOnFirstLayer(): Boolean {
        return tcfSettings?.hideNonIabOnFirstLayer ?: false
    }

    override fun getResurfacePurposeChanged(): Boolean {
        val resurfacePurposeChanged = tcfSettings!!.resurfacePurposeChanged
        return resurfacePurposeChanged && !getSelectedTCFVendors().all {
            val disclosedData = disclosedVendorsMap[it.id] ?: return@all false
            disclosedData.contains(it.toStorageVendor())
        }
    }

    override fun getResurfacePeriodEnded(): Boolean {
        if (tcfSettings!!.resurfacePeriodEnded) {
            storageInstance.lastInteractionTimestamp()
        }
        return tcfSettings!!.resurfacePeriodEnded
    }

    override fun getResurfaceVendorAdded(): Boolean {
        val resurfaceVendorAdded = tcfSettings!!.resurfaceVendorAdded
        return resurfaceVendorAdded && !getSelectedTCFVendors().map { it.id }.all {
            disclosedVendorsMap.keys.contains(it)
        }
    }

    override fun getResurfaceATPChanged(): Boolean {
        val resurfaceATPListChanged = tcfSettings?.resurfaceATPListChanged

        if (resurfaceATPListChanged == false) {
            return false
        }

        return additionalConsentModeService.didATPSChange(tcfSettings?.selectedATPIds?.sorted() ?: emptyList())
    }

    // FIXME: improve this performance
    private fun getPurposeIdsFromVendorsAndStacks(): List<Int> {
        val purposeIdsFromVendors = mutableListOf<Int>()
        val purposeIdsFromStacks = mutableListOf<Int>()

        getVendors().forEach { vendor ->
            purposeIdsFromVendors.addAll(vendor.purposes.map { purpose -> purpose.id })
            purposeIdsFromVendors.addAll(vendor.legitimateInterestPurposes.map { purpose -> purpose.id })
        }

        this.getStacks().forEach { stack ->
            purposeIdsFromStacks.addAll(stack.purposeIds)
        }

        val purposeIdsFromVendorsAndStack: MutableList<Int> = mutableListOf()
        purposeIdsFromVendorsAndStack.addAll(purposeIdsFromVendors)
        purposeIdsFromVendorsAndStack.addAll(purposeIdsFromStacks)

        val purposesFlatlyNotAllowed = this.changedPurposes.notAllowedPurposes
        val purposes = purposeIdsFromVendorsAndStack.distinct().filter { !purposesFlatlyNotAllowed.contains(it) }

        if (tcfSettings!!.purposeOneTreatment) {
            return purposes.drop(1)
        }
        return purposes.toList()
    }

    private fun getSpecialFeatureIdsFromVendorsAndStacks(): List<Int> {
        val tcf2Settings = tcfSettings!!
        val specialFeatureIdsFromVendors: MutableList<Int> = mutableListOf()
        val specialFeatureIdsFromStacks: MutableList<Int> = mutableListOf()

        getVendors().forEach { vendor ->
            specialFeatureIdsFromVendors.addAll(
                vendor.specialFeatures
                    .filter { specialFeature ->
                        !tcf2Settings.disabledSpecialFeatures.contains(
                            specialFeature.id
                        )
                    }
                    .map { element -> element.id }
            )
        }

        this.getStacks().forEach { stack ->
            specialFeatureIdsFromStacks.addAll(
                stack.specialFeatureIds.filter { specialFeatureId ->
                    !tcf2Settings.disabledSpecialFeatures.contains(
                        specialFeatureId
                    )
                }
            )
        }

        val specialFeatureIdsFromVendorsAndStacks = mutableListOf<Int>()

        specialFeatureIdsFromVendorsAndStacks.addAll(specialFeatureIdsFromVendors)
        specialFeatureIdsFromVendorsAndStacks.addAll(specialFeatureIdsFromStacks)

        return specialFeatureIdsFromVendorsAndStacks.distinct()
    }

    override fun getGdprAppliesOnTCF(): Boolean {
        if (isRulesetMarkedNoShow()) {
            return false
        }

        val gdprAppliesOnlyEU = tcfSettings?.gdprApplies ?: false
        val isUserInEU = locationService.location.isInEU()

        return !gdprAppliesOnlyEU || isUserInEU
    }

    private fun isRulesetMarkedNoShow(): Boolean {
        return settingsOrchestrator.noShow
    }

    override fun updateIABTCFKeys(tcString: String) {
        assertNotUIThread()

        val safeTCModel = tcModel ?: return
        val gdprApplies = if (getGdprAppliesOnTCF()) 1 else 0

        val keys = TCFKeysEncoder(
            tcModel = safeTCModel,
            tcString = tcString,
            gdprApplies = gdprApplies
        ).encode()

        val tcfStoragePayload = keys.saveKeys()
        storageInstance.storeValuesDefaultStorage(tcfStoragePayload.values)
    }

    // VISIBLE FOR TESTING
    fun getTCStringFromModel(): String = TCString.encode(tcModel!!)

    private fun updatePolicyVersion() {
        if (this.tcModel!!.getPolicyVersion() != Constants.TCF_POLICY_VERSION) {
            this.tcModel!!.setPolicyVersion(StringOrNumber.Int(Constants.TCF_POLICY_VERSION))
        }
    }

    private fun updateTCString(fromLayer: TCFDecisionUILayer) {
        try {
            this.tcModel?.setConsentScreen(StringOrNumber.Int(fromLayer.value))
            this.tcModel?.setCreatedAndUpdatedFields()
            resetTCFData()

            dispatcher.dispatch {
                semaphore.acquire()

                updatePolicyVersion()

                val updatedTCString = getTCStringFromModel()
                updateIABTCFKeys(tcString = updatedTCString)

                storageInstance.saveTCFData(
                    tcfData = StorageTCF(
                        vendorsDisclosedMap = disclosedVendorsMap,
                        tcString = updatedTCString,
                        acString = additionalConsentModeService.acString,
                    )
                )

                setTCFData()
            }.onSuccess {
                consentsService.saveConsentsState(UsercentricsConsentAction.TCF_STRING_CHANGE)
                semaphore.release()
            }.onFailure {
                logger.error("Failed while trying to updateTCString method", it)
                semaphore.release()
            }
        } catch (ex: Exception) {
            throw ex
        } finally {
        }
    }

    override fun setCmpId(id: Int) {
        dispatcher.dispatch {
            tcModel?.setCmpId(StringOrNumber.Int(id))
            val updatedTCString = getTCStringFromModel()
            updateIABTCFKeys(tcString = updatedTCString)
        }.onFailure {
            logger.error("Failed while trying to setCmpId method", it)
        }
    }

    override fun updateChoices(decisions: TCFUserDecisions, fromLayer: TCFDecisionUILayer) {
        runCatching {
            val tcf2Settings = tcfSettings!!
            val mergedDecisions = createTCFUserDecisionsMergingWithCurrentData(decisions)

            if (mergedDecisions.purposes != null) {
                savePurposes(mergedDecisions.purposes)
            }

            if (mergedDecisions.specialFeatures != null) {
                saveSpecialFeatures(mergedDecisions.specialFeatures)
            }

            if (mergedDecisions.vendors != null) {
                saveVendors(mergedDecisions.vendors)
            }

            setDisclosedVendors(tcfSettings!!, getVendors().toStorageVendorMap())

            // Need to untoggle manually legitimate interest on purposes/vendors to remove previously given consents
            if (tcf2Settings.hideLegitimateInterestToggles) {
                tcModel!!.unsetAllVendorLegitimateInterests()
                tcModel!!.unsetAllPurposeLegitimateInterests()
            }

            if (mergedDecisions.purposes != null || mergedDecisions.specialFeatures != null || mergedDecisions.vendors != null) {
                updateTCString(fromLayer)
            }
        }.onFailure {
            logger.error("Something went wrong with TCF updateChoices method: $it", it)
        }
    }

    private fun createTCFUserDecisionsMergingWithCurrentData(decisions: TCFUserDecisions): TCFUserDecisions {
        val decisionsPurposes = decisions.purposes ?: emptyList()
        val decisionsVendors = decisions.vendors ?: emptyList()

        val purposesData = mapToIdAndConsent(
            items = purposes,
            getId = { it.id },
            showConsentToggle = { it.showConsentToggle },
            showLegitimateInterestToggle = { it.showLegitimateInterestToggle },
            getConsent = { it.consent },
            getLegitimateInterestConsent = { it.legitimateInterestConsent }
        )

        val vendorsData = mapToIdAndConsent(
            items = vendors,
            getId = { it.id },
            showConsentToggle = { it.showConsentToggle },
            showLegitimateInterestToggle = { it.showLegitimateInterestToggle },
            getConsent = { it.consent },
            getLegitimateInterestConsent = { it.legitimateInterestConsent },
            implicitLegitimateInterest = { vendor ->
                hasImplicitLegitimateInterest(vendor.purposes, vendor.legitimateInterestPurposes, vendor.specialPurposes)
            }
        )

        val mergedPurposes = mergeConsentsWithUserDecisions(purposesData, decisionsPurposes)
        val mergedVendors = mergeConsentsWithUserDecisions(vendorsData, decisionsVendors)

        return TCFUserDecisions(
            purposes = mergedPurposes.map { TCFUserDecisionOnPurpose(it.id, consent = it.consent, legitimateInterestConsent = it.legitimateInterestConsent) },
            vendors = mergedVendors.map { TCFUserDecisionOnVendor(it.id, consent = it.consent, legitimateInterestConsent = it.legitimateInterestConsent) },
            specialFeatures = decisions.specialFeatures
        )
    }

    private fun <T> mapToIdAndConsent(
        items: List<T>,
        getId: (T) -> Int,
        showConsentToggle: (T) -> Boolean,
        showLegitimateInterestToggle: (T) -> Boolean,
        getConsent: (T) -> Boolean?,
        getLegitimateInterestConsent: (T) -> Boolean?,
        implicitLegitimateInterest: (T) -> Boolean = { false }
    ): List<IdAndConsent> {

        return items.map { item ->

            val legitimateInterestConsent =
                if (implicitLegitimateInterest(item)) {
                    true
                } else if (showLegitimateInterestToggle(item)) {
                    getLegitimateInterestConsent(item) ?: PredefinedUIDecision.DEFAULT_LEGITIMATE_INTEREST_VALUE
                } else null

            IdAndConsent(
                id = getId(item),
                consent = if (showConsentToggle(item)) {
                    getConsent(item) ?: PredefinedUIDecision.DEFAULT_CONSENT_VALUE
                } else {
                    null
                },
                legitimateInterestConsent = legitimateInterestConsent
            )
        }
    }

    private fun mergeConsentsWithUserDecisions(data: List<IdAndConsent>, decisions: List<TCFConsentWithLegitimateInterestDecision>): List<TCFConsentDecision> {
        val mergedDecisions = mutableListOf<TCFConsentDecision>()

        data.forEach { consentModel ->
            val userDecision = decisions.firstOrNull { it.id == consentModel.id }

            mergedDecisions.add(
                TCFConsentDecision(
                    id = consentModel.id,
                    consent = userDecision?.consent ?: consentModel.consent,
                    legitimateInterestConsent = userDecision?.legitimateInterestConsent ?: consentModel.legitimateInterestConsent
                )
            )
        }
        return mergedDecisions
    }

    private fun getFeaturesFromVendors(): List<TCFFeature> {
        val featureIds: List<Int> = this.getFeatureIdsFromVendors()
        val featuresFromVendorsList: MutableList<TCFFeature> = mutableListOf()

        featureIds.forEach { featureId ->
            val feature = this.tcModel?.getGvl()?.features?.get(featureId.toString())

            if (feature != null) {
                featuresFromVendorsList.add(
                    TCFFeature(
                        purposeDescription = feature.description,
                        illustrations = feature.illustrations,
                        id = feature.id,
                        name = feature.name
                    )
                )
            }
        }
        return featuresFromVendorsList.toList()
    }

    private fun getFeatureIdsFromVendors(): List<Int> {
        val featureIdsFromVendorsList: MutableList<Int> = mutableListOf()

        this.getVendors().forEach { vendor ->
            featureIdsFromVendorsList.addAll(vendor.features.map { feature -> feature.id })
        }

        return featureIdsFromVendorsList.distinct()
    }

    private fun getPurposesFromVendors(): List<TCFPurpose> {
        if (this.purposes.isEmpty()) {
            this.setPurposes()
        }
        return this.purposes.toList()
    }

    private fun setPurposes() {
        val purposeIds: List<Int> = this.getPurposeIdsFromVendorsAndStacks()
        val stacks: List<TCFStack> = this.getStacks()
        val vendors = this.getVendors()
        val tcf2Settings = tcfSettings!!

        var vendorsPurposes = mutableListOf<Int>()
        var vendorsLegitimateInterestPurposes = mutableListOf<Int>()
        val purposesFromVendors = mutableListOf<TCFPurpose>()

        vendors.map { vendor ->
            vendor.legitimateInterestPurposes.map { legIntPurpose -> legIntPurpose.id }
        }.forEach { list ->
            vendorsLegitimateInterestPurposes.addAll(list)
        }

        vendorsLegitimateInterestPurposes = vendorsLegitimateInterestPurposes.distinct().toMutableList()

        vendors.map { vendor ->
            vendor.purposes.map { purpose -> purpose.id }
        }.forEach { list ->
            vendorsPurposes.addAll(list)
        }

        vendorsPurposes = vendorsPurposes.distinct().toMutableList()

        purposeIds.forEach { purposeId ->
            val purpose = this.tcModel?.getGvl()?.purposes?.get(purposeId.toString())
            val stackIncludingPurpose = stacks.find { stack ->
                stack.purposeIds.contains(purposeId)
            }

            if (purpose != null) {
                purposesFromVendors.add(
                    TCFPurpose(
                        consent = this.tcModel?.purposeConsents?.has(purposeId),
                        purposeDescription = purpose.description,
                        id = purpose.id,
                        isPartOfASelectedStack = stackIncludingPurpose != null,
                        legitimateInterestConsent = if (disclosedVendorsMap.isNotEmpty()) {
                            this.tcModel?.purposeLegitimateInterests?.has(purposeId)
                        } else {
                            null
                        },
                        name = purpose.name,
                        showConsentToggle = vendorsPurposes.contains(purposeId) && tcf2Settings.useGranularChoice,
                        showLegitimateInterestToggle = purpose.id != PURPOSE_ONE_ID &&
                                vendorsLegitimateInterestPurposes.contains(purposeId) &&
                                tcf2Settings.useGranularChoice &&
                                !tcf2Settings.hideLegitimateInterestToggles,
                        stackId = stackIncludingPurpose?.id,
                        illustrations = purpose.illustrations,
                        numberOfVendors = getNumberOfVendorsPerPurpose(vendors, purposeId)
                    )
                )
            }
        }

        this.purposes.apply {
            // TODO ATOMIC
            clear()
            addAll(purposesFromVendors.toList().sortedAlphaBy { it.name })
        }
    }

    private fun getNumberOfVendorsPerPurpose(vendors: List<TCFVendor>, purposeId: Int): Int {
        return vendors.sumOf { vendor -> (vendor.purposes + vendor.legitimateInterestPurposes).count { it.id == purposeId } }
    }

    private fun getSpecialFeaturesFromVendorsAndStacks(): List<TCFSpecialFeature> {

        val specialFeatureIds: List<Int> = this.getSpecialFeatureIdsFromVendorsAndStacks()
        val stacks: List<TCFStack> = this.getStacks()
        val tcf2Settings = tcfSettings!!
        val specialFeaturesFromVendorsAndStacks = mutableListOf<TCFSpecialFeature>()

        specialFeatureIds.forEach { specialFeatureId ->

            val specialFeature =
                this.tcModel?.getGvl()?.specialFeatures?.get(specialFeatureId.toString())
            val stackIncludingSpecialFeature =
                stacks.find { stack -> stack.specialFeatureIds.contains(specialFeatureId) }

            if (specialFeature != null) {
                specialFeaturesFromVendorsAndStacks.add(
                    TCFSpecialFeature(
                        consent = this.tcModel?.specialFeatureOptins?.has(specialFeatureId),
                        purposeDescription = specialFeature.description,
                        illustrations = specialFeature.illustrations,
                        id = specialFeature.id,
                        isPartOfASelectedStack = stackIncludingSpecialFeature != null,
                        name = specialFeature.name,
                        showConsentToggle = tcf2Settings.useGranularChoice,
                        stackId = stackIncludingSpecialFeature?.id
                    )
                )
            }
        }

        return specialFeaturesFromVendorsAndStacks.toList()
    }

    private fun getSpecialPurposesFromVendors(): List<TCFSpecialPurpose> {
        val specialPurposeIds: List<Int> = this.getSpecialPurposeIdsFromVendors()
        val specialPurposesFromVendors = mutableListOf<TCFSpecialPurpose>()

        specialPurposeIds.forEach { specialPurposeId ->

            val specialPurpose =
                this.tcModel?.getGvl()?.specialPurposes?.get(specialPurposeId.toString())

            if (specialPurpose != null) {
                specialPurposesFromVendors.add(
                    TCFSpecialPurpose(
                        purposeDescription = specialPurpose.description,
                        illustrations = specialPurpose.illustrations,
                        id = specialPurpose.id,
                        name = specialPurpose.name
                    )
                )
            }
        }

        return specialPurposesFromVendors.toList()
    }

    private fun getSpecialPurposeIdsFromVendors(): List<Int> {
        val specialPurposeIdsFromVendors = mutableListOf<Int>()

        getVendors().forEach { vendor ->
            specialPurposeIdsFromVendors.addAll(vendor.specialPurposes.map { element -> element.id })
        }

        return specialPurposeIdsFromVendors.distinct()
    }

    // TODO: We should check if we are allowed to change the Purpose/SpecialFeature/Vendor in question (if they have been disclosed)
    // TODO: and all disclosed purposes that we do not get a decision for should be set to false?!
    private fun savePurposes(decisions: List<TCFUserDecisionOnPurpose>) {
        decisions.forEach { purpose ->
            if (purpose.consent == true) {
                this.tcModel?.purposeConsents?.set(purpose.id)
            } else {
                this.tcModel?.purposeConsents?.unset(purpose.id)
            }

            if (purpose.legitimateInterestConsent == true) {
                this.tcModel?.purposeLegitimateInterests?.set(purpose.id)
            } else {
                this.tcModel?.purposeLegitimateInterests?.unset(purpose.id)
            }
        }
    }

    private fun saveSpecialFeatures(decisions: List<TCFUserDecisionOnSpecialFeature>) {
        decisions.forEach { specialFeature ->
            if (specialFeature.consent == true) {
                this.tcModel?.specialFeatureOptins?.set(specialFeature.id)
            } else {
                this.tcModel?.specialFeatureOptins?.unset(specialFeature.id)
            }
        }
    }

    private fun saveVendors(decisions: List<TCFUserDecisionOnVendor>) {
        val internalTCModel = tcModel!!

        val tempSetVendorConsents = mutableListOf<Int>()
        val tempUnsetVendorConsents = mutableListOf<Int>()

        val tempSetLegitimateInterest = mutableListOf<Int>()
        val tempUnsetLegitimateInterest = mutableListOf<Int>()

        decisions.forEach { vendor ->
            if (vendor.consent == true) {
                tempSetVendorConsents.add(vendor.id)
            } else {
                tempUnsetVendorConsents.add(vendor.id)
            }

            if (vendor.legitimateInterestConsent == true) {
                tempSetLegitimateInterest.add(vendor.id)
            } else {
                tempUnsetLegitimateInterest.add(vendor.id)
            }
        }

        internalTCModel.vendorConsents.set(tempSetVendorConsents)
        internalTCModel.vendorConsents.unset(tempUnsetVendorConsents)

        internalTCModel.vendorLegitimateInterests.set(tempSetLegitimateInterest)
        internalTCModel.vendorLegitimateInterests.unset(tempUnsetLegitimateInterest)
    }

    private suspend fun resetGVLWithLanguage(language: String): Result<Unit> {
        return try {
            val gvl = this.tcModel?.getGvl()!!
            val changeLanguageResult = gvl.changeLanguage(language = language)

            val exceptionOnChangingLanguage = changeLanguageResult.exceptionOrNull()
            if (exceptionOnChangingLanguage != null) {
                throw exceptionOnChangingLanguage
            }

            return Result.success(Unit)
        } catch (error: Throwable) {
            Result.failure(UsercentricsException(message = "${TCF_WARN_MESSAGES.RESET_GVL_FAILURE.message}: ${error.message}", cause = error))
        }
    }

    override fun clearTCFConsentsData() {
        resetTCFData()

        this.disclosedVendorsMap.clear()
        this.tcModel?.clearConsents()
    }

    private fun resetTCFData() {
        this.vendors.clear()
        this.purposes.clear()
        this.tcfData = null
        this.vendorsIdsWithImplicitLegitimateInterest.clear()
    }

    /**
     * IAB has its own set of supported languages (available here: https://register.consensu.org/translation)
     * and our CMP has its own set of supported languages
     *
     * This method checks whether the provided language is supported by IAB
     * and if not, returns the fallback language
     * */
    private fun resolveLanguage(language: String): String {
        return ConsentLanguages.getLanguageOrSimilarDialect(language)
    }
}

private fun List<TCFVendor>.toStorageVendorMap(): Map<Int, StorageVendor> {
    return associate { vendor ->
        vendor.id to vendor.toStorageVendor()
    }
}

private fun TCFVendor.toStorageVendor(): StorageVendor {
    return StorageVendor(
        legitimateInterestPurposeIds = legitimateInterestPurposes.map { it.id },
        consentPurposeIds = purposes.map { it.id },
        specialPurposeIds = specialPurposes.map { it.id },
    )
}
