package com.usercentrics.sdk.unity

import com.usercentrics.sdk.*
import com.usercentrics.sdk.models.common.UsercentricsVariant
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.unity.data.*
import com.usercentrics.sdk.unity.provider.UnityMessageProvider
import com.usercentrics.sdk.unity.provider.UnityMessageSender
import kotlin.jvm.JvmStatic

@Suppress("VARIABLE_IN_SINGLETON_WITHOUT_THREAD_LOCAL")
class UsercentricsUnity {

    companion object {

        private const val UC_GAME_OBJECT_NAME = "Usercentrics"
        private const val INVALID_PARAMETER = "You have passed an null or empty argument."

        private val jsonParser = UnityJsonParser()

        private val unityMessageSender: UnityMessageSender
            get() = unityMessageProvider.provide()

        internal var unityMessageProvider = UnityMessageProvider()

        private val usercentricsInstance: UsercentricsSDK
            get() = usercentricsInstanceProvider.provide()

        internal var usercentricsInstanceProvider: IUnityUsercentricsInstanceProvider = UnityUsercentricsInstanceProvider()

        @JvmStatic
        fun init(appContext: UsercentricsContext?, rawUnityUserOptions: String) {
            val usercentricsOptions: UsercentricsOptions?
            try {
                usercentricsOptions = jsonParser.decodeFromString(rawUnityUserOptions)!!
            } catch (e: Exception) {
                e.printStackTrace()
                sendInitErrorMessage(INVALID_PARAMETER)
                return
            }

            UsercentricsInternal.initialize(usercentricsOptions, appContext)

            UsercentricsInternal.isReady({ readyStatus ->
                val unityReadyStatus = UnityReadyStatus(readyStatus = readyStatus, settingsId = usercentricsOptions.settingsId)
                sendInitSuccessMessage(jsonParser.encodeToString(unityReadyStatus))
            }, {
                sendInitErrorMessage(it.exception.toString())
            })
        }

        @JvmStatic
        fun getControllerId(): String {
            return usercentricsInstance.getControllerId()
        }

        @JvmStatic
        fun getTCFData() {
            usercentricsInstance.getTCFData {
                sendTCFDataMessage(jsonParser.encodeToString(UnityTCFData(it)))
            }
        }

        @JvmStatic
        fun getUSPData(): String {
            val ccpaData = usercentricsInstance.getUSPData()

            return jsonParser.encodeToString(
                UnityCCPAData(
                    version = ccpaData.version,
                    noticeGiven = ccpaData.noticeGiven,
                    optedOut = ccpaData.optedOut,
                    lspact = ccpaData.lspact,
                    uspString = ccpaData.uspString
                )
            )
        }

        @JvmStatic
        fun restoreUserSession(controllerId: String) {
            usercentricsInstance.restoreUserSession(controllerId, { readyStatus ->
                // FIXME: replace this empty settingsId
                sendRestoreSuccessMessage(jsonParser.encodeToString(UnityReadyStatus(readyStatus = readyStatus, settingsId = "")))
            }, {
                sendRestoreErrorMessage(it.exception.toString())
            })
        }

        @JvmStatic
        fun handleBannerResponse(predefinedUIResponse: UnityPredefinedUIResponse?) {
            sendBannerResponse(jsonParser.encodeToString(predefinedUIResponse))
        }

        @JvmStatic
        fun subscribeOnConsentUpdated() {
            UsercentricsEvent.onConsentUpdated { updatedConsentPayload ->
                sendOnConsentUpdatedMessage(jsonParser.encodeToString(UnityUpdatedConsentPayload(updatedConsentPayload)))
            }
        }

        @JvmStatic
        fun disposeOnConsentUpdatedSubscription() {
            UsercentricsEvent.updatedConsentEvent.tearDown()
        }

        @JvmStatic
        fun subscribeOnConsentMediation() {
            UsercentricsEvent.onConsentMediation { mediationPayload ->
                sendOnConsentMediationMessage(jsonParser.encodeToString(mediationPayload))
            }
        }

        @JvmStatic
        fun disposeOnConsentMediationSubscription() {
            UsercentricsEvent.mediationConsentEvent.tearDown()
        }

        @JvmStatic
        fun getFirstLayerSettings(): String? {
            val activeVariant = usercentricsInstance.getCMPData().activeVariant
            if (activeVariant != UsercentricsVariant.TCF) {
                return null
            }

            val tcfSettings = usercentricsInstance.getCMPData().settings.tcf2 ?: return null

            return jsonParser.encodeToString(
                UnityFirstLayerSettings(
                    title = tcfSettings.firstLayerTitle,
                    description = tcfSettings.firstLayerDescription,
                    additionalInfo = tcfSettings.firstLayerAdditionalInfo,
                    resurfaceNote = tcfSettings.firstLayerNoteResurface,
                    vendorListLinkTitle = tcfSettings.linksVendorListLinkLabel,
                    manageSettingsLinkTitle = tcfSettings.linksManageSettingsLabel,
                    purposesLabel = tcfSettings.labelsPurposes,
                    featuresLabel = tcfSettings.labelsFeatures,
                    acceptAllButton = tcfSettings.buttonsAcceptAllLabel,
                    denyAllButton = tcfSettings.buttonsDenyAllLabel,
                    saveButton = tcfSettings.buttonsSaveLabel,
                )
            )
        }

        @JvmStatic
        fun acceptAllFirstLayerForTCF() {
            usercentricsInstance.acceptAllForTCF(TCFDecisionUILayer.FIRST_LAYER, UsercentricsConsentType.EXPLICIT)
        }

        @JvmStatic
        fun denyAllFirstLayerForTCF() {
            usercentricsInstance.denyAllForTCF(TCFDecisionUILayer.FIRST_LAYER, UsercentricsConsentType.EXPLICIT)
        }

        @JvmStatic
        fun denyAllFirstLayerForGDPR() {
            usercentricsInstance.denyAll(consentType = UsercentricsConsentType.EXPLICIT)
        }

        @JvmStatic
        fun acceptAllFirstLayerForGDPR() {
            usercentricsInstance.acceptAll(consentType = UsercentricsConsentType.EXPLICIT)
        }

        @JvmStatic
        fun saveDecisionsForGDPR(decisions: String): String {
            var decisionsData: List<UserDecision>
            try {
                decisionsData = jsonParser.decodeFromString<UserDecisionWrapper>(decisions).decisions
            } catch (e: Exception) {
                e.printStackTrace()
                sendInitErrorMessage(INVALID_PARAMETER)
                return ""
            }
            val savedDecisions = usercentricsInstance.saveDecisions(decisions = decisionsData, consentType = UsercentricsConsentType.EXPLICIT)
            val data = jsonParser.encodeToString(UsercentricsServiceConsentWrapper(savedDecisions))
            return data
        }

        @JvmStatic
        fun track(eventTypeEnumIndex: Int) {
            val eventType = UsercentricsAnalyticsEventType.values()[eventTypeEnumIndex]
            usercentricsInstance.track(eventType)
        }

        @JvmStatic
        fun setCmpId(cmpId: Int) {
            usercentricsInstance.setCMPId(cmpId)
        }

        @JvmStatic
        fun getCmpData(): String {
            val cmpData = usercentricsInstance.getCMPData()
            return jsonParser.encodeToString(
                UnityCmpData(
                    activeVariant = cmpData.activeVariant,
                    publishedApps = cmpData.settings.publishedApps,
                    userLocation = cmpData.userLocation
                )
            )
        }

        @JvmStatic
        fun setABTestingVariant(variantName: String) {
            usercentricsInstance.setABTestingVariant(variantName)
        }

        @JvmStatic
        fun getABTestingVariant(): String? {
            return usercentricsInstance.getABTestingVariant()
        }

        @JvmStatic
        fun getAdditionalConsentModeData(): String {
            val acmData = usercentricsInstance.getAdditionalConsentModeData()
            return jsonParser.encodeToString(
                UnityAdditionalConsentModeData(
                    acString = acmData.acString,
                    adTechProviders = acmData.adTechProviders
                )
            )
        }

        @JvmStatic
        fun getConsents(): String {
            val consents = usercentricsInstance.getConsents()
            return jsonParser.encodeToString(
                UnityConsents(consents = consents.map { UnityServiceConsent(it) })
            )
        }

        @JvmStatic
        fun clearUserSession() {
            usercentricsInstance.clearUserSession({ readyStatus ->
                // FIXME: replace this empty settingsId
                sendClearSuccessMessage(jsonParser.encodeToString(UnityReadyStatus(readyStatus = readyStatus, settingsId = "")))
            }, {
                sendClearErrorMessage(it.exception.toString())
            })
        }

        private fun sendInitSuccessMessage(readyStatusString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleInitSuccess", readyStatusString)
        }

        private fun sendInitErrorMessage(errorMessage: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleInitError", errorMessage)
        }

        private fun sendBannerResponse(predefinedUIResponseString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleBannerResponse", predefinedUIResponseString)
        }

        private fun sendRestoreSuccessMessage(readyStatusString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleRestoreSuccess", readyStatusString)
        }

        private fun sendRestoreErrorMessage(errorMessage: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleRestoreError", errorMessage)
        }

        private fun sendClearSuccessMessage(readyStatusString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleClearSuccess", readyStatusString)
        }

        private fun sendClearErrorMessage(errorMessage: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleClearError", errorMessage)
        }

        private fun sendTCFDataMessage(tcfData: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleTCFData", tcfData)
        }

        private fun sendOnConsentUpdatedMessage(updatedConsentEventString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleOnConsentUpdated", updatedConsentEventString)
        }

        private fun sendOnConsentMediationMessage(mediationEventString: String) {
            unityMessageSender.sendUnityMessage(UC_GAME_OBJECT_NAME, "HandleOnConsentMediation", mediationEventString)
        }
    }
}
