package com.usercentrics.sdk.v2.banner.service.mapper.ccpa

import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.UsercentricsMaps
import com.usercentrics.sdk.extensions.isMultiple
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.v2.banner.service.mapper.GenericSecondLayerMapper
import com.usercentrics.sdk.v2.banner.service.mapper.FooterButtonLayoutMapper
import com.usercentrics.sdk.v2.banner.service.mapper.PoweredByMapper
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization

internal class CCPASecondLayerMapper(
    private val settings: UsercentricsSettings,
    private val customization: PredefinedUICustomization,
    private val controllerId: String,
    private val categories: List<UsercentricsCategory>,
    private val services: List<LegacyService>,
    private val optOutToggleInitialValue: Boolean,
    private val translations: LegalBasisLocalization
): GenericSecondLayerMapper(settings) {

    companion object {
        private val defaultLogoPosition = FirstLayerLogoPosition.LEFT
    }

    fun map(): UCUISecondLayerSettings {
        return UCUISecondLayerSettings(
            headerSettings = header(),
            footerSettings = footer(),
            contentSettings = content()
        )
    }

    private fun header(): PredefinedUIHeaderSettings {
        return PredefinedUISecondLayerHeaderSettings(
            contentDescription = settings.ccpa!!.secondLayerDescription,
            title = settings.ccpa.secondLayerTitle,
            logoPosition = defaultLogoPosition,
            language = headerLanguageSettings(),
            logoURL = settings.customization?.logoUrl,
            links = headerLinks(),
            firstLayerCloseLink = null,
            firstLayerCloseIcon = null,
        )
    }

    private fun headerLanguageSettings(): PredefinedUILanguageSettings? {
        val availableLanguages = settings.languagesAvailable
            .map { PredefinedUILanguage(it) }
            .sortedBy { it.fullName }
        val hideLanguage = (settings.ccpa!!.secondLayerHideLanguageSwitch) || !availableLanguages.isMultiple()
        return if (hideLanguage) {
            null
        } else {
            PredefinedUILanguageSettings(availableLanguages, PredefinedUILanguage(settings.language))
        }
    }

    private fun headerLinks(): List<PredefinedUILink> {
        return listOf(
            PredefinedUILink.legalLinkUrl(
                label = settings.labels.privacyPolicyLinkText,
                url = settings.privacyPolicyUrl,
                eventType = UsercentricsAnalyticsEventType.PRIVACY_POLICY_LINK,
            ),
            PredefinedUILink.legalLinkUrl(
                label = settings.labels.imprintLinkText,
                url = settings.imprintUrl,
                eventType = UsercentricsAnalyticsEventType.IMPRINT_LINK,
            )
        ).filter { !it.isEmpty() }
    }

    private fun footer(): PredefinedUIFooterSettings {
        val optOutToggle = if (!settings.ccpa!!.removeDoNotSellToggle) {
            PredefinedUIFooterEntry(settings.ccpa.optOutNoticeLabel)
        } else {
            null
        }
        val poweredBy = LegacyPoweredBy(
            isEnabled = settings.enablePoweredBy,
        )
        val buttonLayoutMapper = FooterButtonLayoutMapper(
            okButton = PredefinedUIFooterButton(
                label = settings.ccpa.btnSave,
                customization = customization.color.okButton,
                type = PredefinedUIButtonType.OK
            ),
        )
        return PredefinedUIFooterSettings(
            poweredBy = PoweredByMapper.mapPoweredBy(poweredBy),
            optOutToggle = optOutToggle,
            optOutToggleInitialValue = optOutToggleInitialValue,
            buttons = buttonLayoutMapper.mapButtons(),
            buttonsLandscape = buttonLayoutMapper.mapButtonsLandscape(),
        )
    }

    private fun content(): List<PredefinedUITabSettings> {
        return listOf(
            PredefinedUITabSettings(
                title = settings.secondLayer.tabsCategoriesLabel,
                content = PredefinedUICategoriesContent(cardUISections = listOf(contentCategorySection()))
            ),
            PredefinedUITabSettings(
                title = settings.secondLayer.tabsServicesLabel,
                content = PredefinedUIServicesContent(cardUISections = listOf(contentServiceSection()))
            )
        )
    }

    private fun contentCategorySection(): PredefinedUICardUISection {
        val categoriesEntries = UsercentricsMaps.mapCategories(categories, services)
        return PredefinedUICardUISection(
            title = null,
            cards = categoriesEntries
                .map { entry ->
                    PredefinedUICardUI(
                        entry = entry,
                        mainSwitchSettings = null,
                        content = PredefinedUIServicesCardContent(
                            services = entry.services.map {
                                PredefinedUIServiceDetails(
                                    service = it,
                                    mainSwitchSettings = null,
                                    dpsDisplayFormat = settings.dpsDisplayFormat,
                                    consent = predefinedUIServiceConsent(it.consent),
                                )
                            },
                        ),
                        shortDescription = entry.category.description
                    )
                }
        )
    }

    private fun contentServiceSection(): PredefinedUICardUISection {
        val servicesToShow = services.filter { !it.isHidden }
        return PredefinedUICardUISection(
            title = null,
            cards = servicesToShow.map { service ->

                PredefinedUICardUI(
                    service = service,
                    content = PredefinedUISingleServiceCardContent(
                        service = PredefinedUIServiceDetails(
                            service = service,
                            mainSwitchSettings = null,
                            dpsDisplayFormat = settings.dpsDisplayFormat,
                            consent = predefinedUIServiceConsent(service.consent),
                        ),
                    ),
                    mainSwitchSettings = null,
                )
            },
            controllerID = PredefinedUIControllerIDSettings(
                label = translations.labels.controllerIdTitle,
                value = controllerId
            )
        )
    }
}
