package com.usercentrics.sdk.v2.banner.service.mapper.gdpr

import com.usercentrics.sdk.CategoryProps
import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.UsercentricsMaps
import com.usercentrics.sdk.extensions.emptyToNull
import com.usercentrics.sdk.extensions.isMultiple
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.v2.banner.service.mapper.GenericSecondLayerMapper
import com.usercentrics.sdk.v2.banner.service.mapper.FooterButtonLayoutMapper
import com.usercentrics.sdk.v2.banner.service.mapper.PoweredByMapper
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization

internal class GDPRSecondLayerMapper(
    private val settings: UsercentricsSettings,
    private val translations: LegalBasisLocalization,
    private val customization: PredefinedUICustomization,
    private val controllerId: String,
    private val categories: List<UsercentricsCategory>,
    private val services: List<LegacyService>,
    private val serviceLabels: PredefinedUIServiceLabels,
) : GenericSecondLayerMapper(settings) {

    companion object {
        private val defaultLogoPosition = FirstLayerLogoPosition.LEFT
    }

    private val hideTogglesForServices = settings.secondLayer.hideTogglesForServices
    private val hideDataProcessingServices = settings.secondLayer.hideDataProcessingServices

    fun map(): UCUISecondLayerSettings {
        return UCUISecondLayerSettings(
            headerSettings = header(),
            footerSettings = footer(),
            contentSettings = content()
        )
    }

    fun mapTV(): PredefinedTVSecondLayerSettings {
        return PredefinedTVSecondLayerSettings(
            title = settings.labels.secondLayerTitle ?: "",
            save = settings.labels.btnSave,
            toggleLocalizations = PredefinedUIToggleLocalizations(
                onText = settings.tcf2?.togglesSpecialFeaturesToggleOn ?: "On",
                offText = settings.tcf2?.togglesSpecialFeaturesToggleOff ?: "Off"
            ),
            content = PredefinedTVSecondLayerSettingsContent.Cards(
                listOf(
                    GDPRSectionMapperTV(
                        categoriesSectionLabel = settings.secondLayer.tabsCategoriesLabel,
                        servicesSectionLabel = settings.secondLayer.tabsServicesLabel,
                        consentLabel = settings.labels.consent,
                        hideTogglesForServices = hideTogglesForServices,
                        settings = settings,
                        serviceLabels = serviceLabels,
                        translations = translations
                    ).map(UsercentricsMaps.mapCategories(categories, services))
                )
            ),
            denyAll = denyAllText(),
        )
    }

    private fun header(): PredefinedUIHeaderSettings {
        return PredefinedUISecondLayerHeaderSettings(
            contentDescription = settings.labels.secondLayerDescriptionHtml.emptyToNull() ?: settings.labels.headerModal,
            title = settings.labels.secondLayerTitle ?: "",
            logoPosition = defaultLogoPosition,
            language = headerLanguageSettings(),
            logoURL = settings.customization?.logoUrl,
            links = headerLinks(),
            firstLayerCloseLink = null,
            firstLayerCloseIcon = null,
        )
    }

    private fun headerLanguageSettings(): PredefinedUILanguageSettings? {
        val availableLanguages = settings.languagesAvailable
            .map { PredefinedUILanguage(it) }
            .sortedBy { it.fullName }
        val hideLanguage = (settings.secondLayer.hideLanguageSwitch == true) || !availableLanguages.isMultiple()
        return if (hideLanguage) {
            null
        } else {
            PredefinedUILanguageSettings(availableLanguages, PredefinedUILanguage(settings.language))
        }
    }

    private fun headerLinks(): List<PredefinedUILink> {
        return if (hideDataProcessingServices) {
            emptyList()
        } else {
            listOf(
                PredefinedUILink.legalLinkUrl(
                    label = settings.labels.privacyPolicyLinkText,
                    url = settings.privacyPolicyUrl,
                    eventType = UsercentricsAnalyticsEventType.PRIVACY_POLICY_LINK,
                ),
                PredefinedUILink.legalLinkUrl(
                    label = settings.labels.imprintLinkText,
                    url = settings.imprintUrl,
                    eventType = UsercentricsAnalyticsEventType.IMPRINT_LINK,
                ),
            ).filter { it.label.isNotEmpty() }
        }
    }

    private fun footer(): PredefinedUIFooterSettings {
        val denyAllButton = if (settings.secondLayer.hideButtonDeny == true) {
            null
        } else {
            PredefinedUIFooterButton(
                label = denyAllText(),
                customization = customization.color.denyAllButton,
                type = PredefinedUIButtonType.DENY_ALL
            )
        }

        val acceptAllButton = PredefinedUIFooterButton(
            label = acceptAllText(),
            customization = customization.color.acceptAllButton,
            type = PredefinedUIButtonType.ACCEPT_ALL
        )

        val saveSettingsButton = PredefinedUIFooterButton(
            label = settings.labels.btnSave,
            customization = customization.color.saveButton,
            type = PredefinedUIButtonType.SAVE_SETTINGS
        )
        val buttonLayoutMapper = FooterButtonLayoutMapper(
            acceptAll = acceptAllButton,
            denyAll = denyAllButton,
            saveSettings = saveSettingsButton,
        )
        return PredefinedUIFooterSettings(
            poweredBy = poweredBy(),
            buttons = buttonLayoutMapper.mapButtons(),
            buttonsLandscape = buttonLayoutMapper.mapButtonsLandscape()
        )
    }

    private fun denyAllText(): String {
        if(settings.secondLayer.hideButtonDeny == true){
            return ""
        }
        return if (settings.secondLayer.denyButtonText?.isNotBlank() == true) {
            settings.secondLayer.denyButtonText
        } else {
            settings.labels.btnDeny
        }
    }

    private fun acceptAllText(): String {
        return if (settings.secondLayer.acceptButtonText?.isNotBlank() == true) {
            settings.secondLayer.acceptButtonText
        } else {
            settings.labels.btnAcceptAll
        }
    }

    private fun poweredBy(): PredefinedUIFooterEntry? {
        return PoweredByMapper.mapPoweredBy(
            LegacyPoweredBy(
                isEnabled = settings.enablePoweredBy,
            )
        )
    }

    private fun content(): List<PredefinedUITabSettings> {
        return if (hideDataProcessingServices) {
            listOf(
                PredefinedUITabSettings(
                    title = "",
                    content = PredefinedUICategoriesContent(cardUISections = listOf(contentCategorySection()))
                )
            )
        } else {
            listOf(
                PredefinedUITabSettings(
                    title = settings.secondLayer.tabsCategoriesLabel,
                    content = PredefinedUICategoriesContent(cardUISections = listOf(contentCategorySection()))
                ),
                PredefinedUITabSettings(
                    title = settings.secondLayer.tabsServicesLabel,
                    content = PredefinedUIServicesContent(cardUISections = listOf(contentServiceSection()))
                )
            )
        }
    }

    private fun contentCategorySection(): PredefinedUICardUISection {
        val categoriesEntries = UsercentricsMaps.mapCategories(categories, services)
        val categoriesEntriesMap = categoriesEntries.map { entry -> getPredefinedUICardUI(entry) }

        val controllerID = if (hideDataProcessingServices) {
            getControllerID()
        } else {
            null
        }

        return PredefinedUICardUISection(
            title = null,
            cards = categoriesEntriesMap,
            controllerID = controllerID
        )
    }

    private fun getPredefinedUICardUI(entry: CategoryProps): PredefinedUICardUI {
        val content = if (hideDataProcessingServices) {
            null
        } else {
            PredefinedUIServicesCardContent(
                services = entry.services.map {
                    PredefinedUIServiceDetails(
                        service = it,
                        hideTogglesForServices = hideTogglesForServices,
                        dpsDisplayFormat = settings.dpsDisplayFormat,
                        consent = predefinedUIServiceConsent(it.consent),
                    )
                },
            )
        }

        return PredefinedUICardUI(
            entry = entry,
            content = content,
            shortDescription = entry.category.description
        )
    }

    private fun contentServiceSection(): PredefinedUICardUISection {
        val servicesToShow = services.filter { !it.isHidden }
        return PredefinedUICardUISection(
            title = null,
            cards = servicesToShow.map { service ->
                val mainSwitchSettings = if (hideTogglesForServices) {
                    null
                } else {
                    PredefinedUISwitchSettingsUI(service = service)
                }

                PredefinedUICardUI(
                    service = service,
                    content = PredefinedUISingleServiceCardContent(
                        service = PredefinedUIServiceDetails(
                            service = service,
                            dpsDisplayFormat = settings.dpsDisplayFormat,
                            consent = predefinedUIServiceConsent(service.consent),
                        ),
                    ),
                    mainSwitchSettings = mainSwitchSettings
                )
            },
            controllerID = getControllerID()
        )
    }

    private fun getControllerID(): PredefinedUIControllerIDSettings {
        return PredefinedUIControllerIDSettings(
            label = translations.labels.controllerIdTitle,
            value = controllerId
        )
    }
}
