package com.usercentrics.sdk.v2.consent.api

import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.core.settings.SettingsOrchestrator
import com.usercentrics.sdk.domain.api.http.HttpErrorResponse
import com.usercentrics.sdk.domain.api.http.HttpRequests
import com.usercentrics.sdk.domain.api.http.HttpResponse
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.extensions.millisToSeconds
import com.usercentrics.sdk.services.api.NetworkResolver
import com.usercentrics.sdk.v2.consent.data.ConsentStatus
import com.usercentrics.sdk.v2.consent.data.ConsentStringObjectDto
import com.usercentrics.sdk.v2.consent.data.ConsentsDataDto
import com.usercentrics.sdk.v2.consent.data.GetConsentsData

internal class GetConsentsApiImpl(
    private val requests: HttpRequests,
    private val networkResolver: NetworkResolver,
    private val jsonParser: JsonParser,
    private val settingsOrchestrator: SettingsOrchestrator,
) : GetConsentsApi {

    override fun getUserConsents(controllerId: String, onSuccess: (GetConsentsData) -> Unit, onError: (UsercentricsException) -> Unit) {
        settingsOrchestrator.settingsIdObservable.subscribe { settingsId ->
            requests.get(
                url = createUrl(settingsId = settingsId, controllerId = controllerId),
                headers = null,
                onSuccess = { httpResponse ->
                    onResponse(httpResponse, onError, onSuccess)
                },
                onError = {
                    onError(UsercentricsException("Failed to get user consents: ${it.message}", it))
                }
            )
        }
    }

    private fun onResponse(httpResponse: HttpResponse, onError: (UsercentricsException) -> Unit, onSuccess: (GetConsentsData) -> Unit) {
        if (httpResponse.statusCode != 200) {
            errorResponse(httpResponse, onError)
            return
        }

        try {
            val response = jsonParser.decodeFromString(ConsentsDataDto.serializer(), httpResponse.body)
            onSuccess(response.mapToGetConsentsData(jsonParser))
        } catch (e: Exception) {
            onError(UsercentricsException("Missing necessary fields to restore user session", e))
        }
    }

    private fun errorResponse(httpResponse: HttpResponse, onError: (UsercentricsException) -> Unit) {
        val response = jsonParser.decodeFromString(HttpErrorResponse.serializer(), httpResponse.body)
        onError(UsercentricsException(response.message.orEmpty()))
    }

    private fun createUrl(settingsId: String, controllerId: String): String {
        return "${networkResolver.getConsentsBaseUrl()}/?settingsId=$settingsId&controllerId=$controllerId"
    }
}

internal fun ConsentsDataDto.mapToGetConsentsData(jsonParser: JsonParser): GetConsentsData {
    val consentMetaPayload = runCatching { jsonParser.decodeFromString(ConsentStringObjectDto.serializer(), this.consentMeta!!) }.getOrNull()

    return GetConsentsData(
        consents = this.consents.map { entry ->
            ConsentStatus(
                action = this.action,
                settingsVersion = this.settingsVersion,
                timestampInSeconds = this.timestampInMillis.millisToSeconds(),
                consentStatus = entry.consentStatus,
                consentTemplateId = entry.consentTemplateId,
            )
        }.sortedBy { it.timestampInSeconds },
        consentStringObject = consentMetaPayload?.toConsentStringObject(consentString),
        acString = this.acString ?: "",
    )
}
