package com.usercentrics.sdk.v2.consent.data

import com.usercentrics.sdk.services.deviceStorage.models.StorageVendor
import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.ListSerializer
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

@Serializable
internal data class ConsentStringObject(
    val string: String,
    val tcfVendorsDisclosedMap: Map<Int, StorageVendor> = emptyMap()
)

// This the consentMeta DTO
@Serializable
internal data class ConsentStringObjectDto(
    @SerialName("timestamp")
    val timestampInMillis: Long,
    // The serialization is:
    // [
    //   vendorId,
    //   [legitimateInterestPurposeIds],
    //   [consentPurposeIds],
    //   [specialPurposeIds]
    //  ]
    val vendors: List<List<@Serializable(MetaVendorEntrySerializer::class) Any>>
) {
    companion object {
        fun create(timestampInMillis: Long, consentStringObject: ConsentStringObject?): ConsentStringObjectDto? {
            if (consentStringObject == null) return null
            return ConsentStringObjectDto(
                timestampInMillis = timestampInMillis,
                vendors = consentStringObject.tcfVendorsDisclosedMap.map {
                    listOf(
                        it.key,
                        it.value.legitimateInterestPurposeIds,
                        it.value.consentPurposeIds,
                        it.value.specialPurposeIds,
                    )
                },
            )
        }
    }

    @Suppress("UNCHECKED_CAST")
    internal fun toConsentStringObject(consentString: String?): ConsentStringObject? {
        if (consentString.isNullOrBlank()) return null
        return ConsentStringObject(
            string = consentString,
            tcfVendorsDisclosedMap = runCatching {
                vendors.associate {
                    val vendorId = it[0] as Int
                    val legitimateInterestPurposeIds = it[1] as List<Int>
                    val consentPurposeIds = it[2] as List<Int>
                    val specialPurposeIds = it[3] as List<Int>
                    vendorId to StorageVendor(
                        legitimateInterestPurposeIds = legitimateInterestPurposeIds,
                        consentPurposeIds = consentPurposeIds,
                        specialPurposeIds = specialPurposeIds,
                    )
                }
            }.getOrElse { emptyMap() }
        )
    }
}


internal class MetaVendorEntrySerializer : KSerializer<Any> {
    override val descriptor: SerialDescriptor = Int.serializer().descriptor

    @Suppress("UNCHECKED_CAST")
    override fun serialize(encoder: Encoder, value: Any) {
        when (value) {
            is Int -> encoder.encodeSerializableValue(Int.serializer(), value)
            is List<*> -> encoder.encodeSerializableValue(ListSerializer(Int.serializer()), value as List<Int>)
            else -> throw IllegalStateException("Unexpected vendors array serialization")
        }
    }

    override fun deserialize(decoder: Decoder): Any {
        return runCatching { decoder.decodeInt() }.getOrElse {
            decoder.decodeSerializableValue(ListSerializer(Int.serializer()))
        }
    }
}

