package com.usercentrics.sdk.v2.etag.cache

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.errors.CacheException
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import com.usercentrics.sdk.v2.etag.repository.EtagHolder
import com.usercentrics.sdk.v2.file.IFileStorage

internal class EtagCacheStorage(
    private val fileStorage: IFileStorage,
    private val dispatcher: Dispatcher,
) : IEtagCacheStorage {

    private var identifier: String? = null

    override fun boot(identifier: String) {
        this.identifier = identifier
        checkIfDirtyDirectoriesExist()
    }

    private fun checkIfDirtyDirectoriesExist() {
        dispatcher.dispatch {
            val directoriesStored = fileStorage.ls("")
            directoriesStored?.filter { it != defaultEtagPath() }?.forEach {
                fileStorage.rmdir(it)
            }
        }
    }

    override fun getRawEtagFileName(key: String): String? {
        return fileStorage.ls(etagDirFor(key))?.firstOrNull()?.let { decodeEtagFileName(it) }
    }

    override fun storeFileAndEtag(etagHolder: EtagHolder) {
        val etagDir = etagDirFor(etagHolder.etagKey)
        fileStorage.rmdir(etagDir)
        fileStorage.mkdir(etagDir)

        val etagFileName = encodeEtagFileName(etagHolder.etagValue)

        val cacheMaxAge = DateTime().addSeconds(etagHolder.cacheMaxAge).timestamp()
        val fileRelativePath = "$etagDir/$etagFileName$cacheControlSeparator$cacheMaxAge"
        fileStorage.storeFile(fileRelativePath, etagHolder.responseBody)
    }

    override fun getStoredFile(key: String, etagValue: String): String {
        val etagFileName = encodeEtagFileName(etagValue)
        return fileStorage.getFile("${etagDirFor(key)}/$etagFileName") ?: throw CacheException(key)
    }

    override fun saveOfflineStaging() {
        fileStorage.rmdir(offlineEtagPath())
        fileStorage.copy(defaultEtagPath(), offlineEtagPath())
    }

    override fun removeOfflineStaging() {
        fileStorage.rmdir(offlineEtagPath())
    }

    override fun restoreOfflineStaging() {
        fileStorage.rmdir(defaultEtagPath())
        fileStorage.copy(offlineEtagPath(), defaultEtagPath())
        fileStorage.rmdir(offlineEtagPath())
    }

    private fun etagDirFor(key: String) = "${defaultEtagPath()}/$key"

    private fun defaultEtagPath() = eTagsDir.plus("-").plus(identifier)
    private fun offlineEtagPath() = eTagsOfflineStagingDir.plus("-").plus(identifier)

    // ETAG specification: https://datatracker.ietf.org/doc/html/rfc7232#section-2.3
    private fun encodeEtagFileName(etagValue: String): String {
        return etagValue.removeSurrounding("\"")
    }

    private fun decodeEtagFileName(etagFileName: String): String {
        return "\"$etagFileName\""
    }

    companion object {
        private const val eTagsDir = "etags"
        private const val eTagsOfflineStagingDir = "etags-staging"
        private const val etagLanguageParamSeparator = "-"

        const val cacheControlSeparator = "@#$"
        const val settingsDir = "settings$etagLanguageParamSeparator"
        const val translationsDir = "translations$etagLanguageParamSeparator"
        const val languagesDir = "languages"
        const val tcfVendorListDir = "tcf-vendorlist"
        const val tcfDeclarationsDir = "tcf-declarations$etagLanguageParamSeparator"
        const val aggregatorDir = "aggregator$etagLanguageParamSeparator"
        const val ruleSetDir = "ruleSet"
        const val additionalConsentModeDir = "acp"
    }
}
